<?php
session_start();
require_once '../../config/db.php';

// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Get report_id
$report_id = isset($_GET['report_id']) ? (int)$_GET['report_id'] : 0;
if (!$report_id) {
    $error = "Report ID is missing.";
    file_put_contents('debug.log', "Error at " . date('Y-m-d H:i:s') . ": $error\n", FILE_APPEND);
    header('Location: till_report.php?error=' . urlencode($error));
    exit;
}

// Fetch report data
$query = "SELECT tr.*, b.branch_name FROM till_reports tr LEFT JOIN branches b ON tr.branch_id = b.id WHERE tr.id = ?";
$stmt = $conn->prepare($query);
if (!$stmt) {
    $error = "Failed to prepare query: " . $conn->error;
    file_put_contents('debug.log', "Error at " . date('Y-m-d H:i:s') . ": $error\n", FILE_APPEND);
    header('Location: till_report.php?error=' . urlencode($error));
    exit;
}
$stmt->bind_param('i', $report_id);
$stmt->execute();
$result = $stmt->get_result();

$reports = [];
while ($row = $result->fetch_assoc()) {
    $reports[] = $row;
}
if (empty($reports)) {
    $error = "No records found for Report ID $report_id.";
    file_put_contents('debug.log', "Error at " . date('Y-m-d H:i:s') . ": $error\n", FILE_APPEND);
    header('Location: till_report.php?error=' . urlencode($error));
    exit;
}

// Check for multiple reports with the same report_id
$warning = null;
if (count($reports) > 1) {
    $warning = "Warning: Multiple reports found for Report ID $report_id. Displaying unique till data only. Contact admin to resolve database duplicates.";
    file_put_contents('debug.log', "Warning at " . date('Y-m-d H:i:s') . ": $warning\n", FILE_APPEND);
}

// Calculate totals for each till across all reports
$tills = [];
$seen_tills = [];
$store_name = $reports[0]['store_name'];
$branch_name = $reports[0]['branch_name'] ?? 'Unknown';
$report_date = $reports[0]['report_date'];
$remarks = $reports[0]['remarks'];

foreach ($reports as $report) {
    file_put_contents('debug.log', "Processing Report ID $report_id, Row: " . print_r($report, true) . "\n", FILE_APPEND);
    for ($i = 1; $i <= 5; $i++) {
        $batch = trim($report["batch_$i"] ?? '');
        $cashier = trim($report["cashier_$i"] ?? '');
        $drop1 = (float)($report["drop1_$i"] ?? 0);
        $drop2 = (float)($report["drop2_$i"] ?? 0);
        $drop3 = (float)($report["drop3_$i"] ?? 0);
        $note_1000 = (float)($report["note_1000_$i"] ?? 0);
        $note_500 = (float)($report["note_500_$i"] ?? 0);
        $note_200 = (float)($report["note_200_$i"] ?? 0);
        $note_100 = (float)($report["note_100_$i"] ?? 0);
        $note_50 = (float)($report["note_50_$i"] ?? 0);
        $coins = (float)($report["coins_$i"] ?? 0);
        $mpesa = (float)($report["mpesa_$i"] ?? 0);
        $visa = (float)($report["visa_$i"] ?? 0);
        // Include till if cashier or any numeric field is non-empty
        if ($cashier || $drop1 || $drop2 || $drop3 || $note_1000 || $note_500 || $note_200 || $note_100 || $note_50 || $coins || $mpesa || $visa) {
            if (empty($cashier)) {
                file_put_contents('debug.log', "Till $i in Report ID $report_id: Skipped due to missing Cashier\n", FILE_APPEND);
                continue;
            }

            $total_drop = $drop1 + $drop2 + $drop3;
            $tt_notes = ($note_1000 * 1000) + ($note_500 * 500) + ($note_200 * 200) + ($note_100 * 100) + ($note_50 * 50);
            $total_cash = $tt_notes + $coins + $total_drop;
            $total_all = $total_cash + $mpesa + $visa;

            $till_key = md5(($batch ?: 'NO_BATCH') . '|' . $cashier);
            if (!in_array($till_key, $seen_tills)) {
                $tills[$i] = [
                    'batch_no' => $batch ?: 'N/A',
                    'cashier' => $cashier,
                    'register' => $report["register_$i"] ?? '',
                    'total_drop' => $total_drop,
                    'tt_notes' => $tt_notes,
                    'coins' => $coins,
                    'total_cash' => $total_cash,
                    'mpesa' => $mpesa,
                    'visa' => $visa,
                    'total_all' => $total_all
                ];
                $seen_tills[] = $till_key;
                file_put_contents('debug.log', "Till $i: Added - Batch='$batch', Cashier='$cashier', Total Cash=$total_cash, Total All=$total_all\n", FILE_APPEND);
            } else {
                file_put_contents('debug.log', "Till $i: Skipped duplicate - Batch='$batch', Cashier='$cashier'\n", FILE_APPEND);
            }
        } else {
            file_put_contents('debug.log', "Till $i: Skipped - No data\n", FILE_APPEND);
        }
    }
}

// Calculate grand totals
$total_drop = array_sum(array_column($tills, 'total_drop'));
$total_tt_notes = array_sum(array_column($tills, 'tt_notes'));
$total_coins = array_sum(array_column($tills, 'coins'));
$total_cash = array_sum(array_column($tills, 'total_cash'));
$total_mpesa = array_sum(array_column($tills, 'mpesa'));
$total_visa = array_sum(array_column($tills, 'visa'));
$total_all = array_sum(array_column($tills, 'total_all'));

?>
include '../../includes/layout2.php';

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Till Report - <?= htmlspecialchars($report_id) ?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body { background: #f4f6f8; font-size: 13px; }
    .report-header {
      border-bottom: 2px solid #0056b3;
      margin-bottom: 20px;
      padding: 10px;
      background: #007bff;
      color: #fff;
      border-radius: 4px;
      text-align: center;
    }
    .summary-card {
      padding: 10px;
      border: 1px solid #dee2e6;
      border-radius: 4px;
      background: #fff;
      box-shadow: 0 2px 4px rgba(0,0,0,0.1);
      text-align: center;
    }
    .signature-line {
      border-top: 1px dashed #000;
      margin-top: 50px;
      width: 200px;
    }
    .table thead {
      background: #0056b3;
      color: #fff;
    }
    .table tfoot {
      background: #d1e7dd;
      font-weight: bold;
    }
    .btn-secondary {
      background-color: #007bff;
      border-color: #007bff;
    }
    .btn-secondary:hover {
      background-color: #0056b3;
      border-color: #0056b3;
    }
    @media print {
      .no-print { display: none !important; }
      body { background: #fff; font-size: 12px; }
      .report-header { background: none; color: #000; border-bottom: 2px solid #000; }
      .summary-card { box-shadow: none; border: 1px solid #000; }
      .table tfoot { background: #e9ecef; }
    }
  </style>
</head>
<body>
<div class="container my-4">
  <?php if (isset($_GET['success'])): ?>
    <div class="alert alert-success"><?php echo htmlspecialchars($_GET['success']); ?></div>
  <?php endif; ?>
  <?php if (isset($_GET['error'])): ?>
    <div class="alert alert-danger"><?php echo htmlspecialchars($_GET['error']); ?></div>
  <?php endif; ?>
  <?php if ($warning): ?>
    <div class="alert alert-warning"><?php echo htmlspecialchars($warning); ?></div>
  <?php endif; ?>

  <div class="report-header">
    <h4 class="fw-bold">DAILY TILL REPORT</h4>
    <div>
      <strong>Store:</strong> <?= htmlspecialchars($store_name) ?> | 
      <strong>Branch:</strong> <?= htmlspecialchars($branch_name) ?> | 
      <strong>Date:</strong> <?= htmlspecialchars($report_date) ?> | 
      <strong>Report ID:</strong> <?= htmlspecialchars($report_id) ?>
    </div>
    <?php if ($remarks): ?>
      <div><strong>Remarks:</strong> <?= htmlspecialchars($remarks) ?></div>
    <?php endif; ?>
  </div>

  <div class="row text-center mb-4">
    <div class="col summary-card">
      <strong>Total Cash</strong><br>Ksh <?= number_format($total_cash, 2) ?>
    </div>
    <div class="col summary-card">
      <strong>MPESA</strong><br>Ksh <?= number_format($total_mpesa, 2) ?>
    </div>
    <div class="col summary-card">
      <strong>VISA</strong><br>Ksh <?= number_format($total_visa, 2) ?>
    </div>
    <div class="col summary-card">
      <strong>Total Collected</strong><br><span class="text-success fw-bold">Ksh <?= number_format($total_all, 2) ?></span>
    </div>
  </div>

  <div class="table-responsive">
    <table class="table table-bordered table-striped table-sm">
      <thead>
        <tr>
          <th>Till</th>
          <th>Batch No</th>
          <th>Cashier</th>
          <th>Register</th>
          <th>Total Drop</th>
          <th>TT Notes</th>
          <th>Coins</th>
          <th>Total Cash</th>
          <th>MPESA</th>
          <th>VISA</th>
          <th>Total All</th>
        </tr>
      </thead>
      <tbody>
        <?php if (empty($tills)): ?>
          <tr><td colspan="11" class="text-center">No active tills found for this report.</td></tr>
        <?php else: ?>
          <?php foreach ($tills as $i => $till): ?>
          <tr>
            <td>Till <?= $i ?></td>
            <td><?= htmlspecialchars($till['batch_no']) ?></td>
            <td><?= htmlspecialchars($till['cashier']) ?></td>
            <td><?= htmlspecialchars($till['register']) ?></td>
            <td><?= number_format($total_drop, 2) ?></td>
            <td><?= number_format($till['tt_notes'], 2) ?></td>
            <td><?= number_format($till['coins'], 2) ?></td>
            <td><?= number_format($till['total_cash'], 2) ?></td>
            <td><?= number_format($till['mpesa'], 2) ?></td>
            <td><?= number_format($till['visa'], 2) ?></td>
            <td><?= number_format($till['total_all'], 2) ?></td>
          </tr>
          <?php endforeach; ?>
        <?php endif; ?>
      </tbody>
      <tfoot>
        <tr>
          <td colspan="4">TOTALS</td>
          <td><?= number_format($total_drop, 2) ?></td>
          <td><?= number_format($total_tt_notes, 2) ?></td>
          <td><?= number_format($total_coins, 2) ?></td>
          <td><?= number_format($total_cash, 2) ?></td>
          <td><?= number_format($total_mpesa, 2) ?></td>
          <td><?= number_format($total_visa, 2) ?></td>
          <td><?= number_format($total_all, 2) ?></td>
        </tr>
      </tfoot>
    </table>
  </div>

  <div class="row mt-5">
    <div class="col text-center">
      <div class="signature-line"></div>
      <small>Prepared By (Cash Office)</small>
    </div>
    <div class="col text-center">
      <div class="signature-line"></div>
      <small>Checked By (Accountant)</small>
    </div>
    <div class="col text-center">
      <div class="signature-line"></div>
      <small>Approved By (Manager)</small>
    </div>
  </div>

  <div class="text-end mt-4 no-print">
    <a href="till_report.php" class="btn btn-secondary btn-sm">Back to Form</a>
    <button onclick="window.print()" class="btn btn-secondary btn-sm">🖨️ Print</button>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>