<?php
// Include header
require_once __DIR__ . '/includes/layout2.php';
require_once __DIR__ . '/config/db.php';

// Fetch user info
$user_id = $_SESSION['user_id'] ?? 0;
$user_name = $_SESSION['user_name'] ?? 'User';
$user_role = $_SESSION['role'] ?? 'User';
$user_branch_id = $_SESSION['branch_id'] ?? 0; // Non-admins may have branch restriction

// Get selected branch from GET or default to 'all'
$selected_branch_id = $_GET['branch_id'] ?? 'all';
if (!is_numeric($selected_branch_id) && $selected_branch_id !== 'all') {
    $selected_branch_id = 'all'; // Sanitize invalid branch_id
}
if ($user_role !== 'Admin' && $selected_branch_id !== 'all' && $selected_branch_id != $user_branch_id) {
    $selected_branch_id = $user_branch_id; // Restrict non-admins to their branch
}

// Fetch branches
$branches_query = "SELECT id, branch_name FROM branches ORDER BY branch_name";
$branches_result = $conn->query($branches_query);
if ($branches_result === false) {
    error_log("Branches Query failed: " . $conn->error);
    die("Database error. Please try again later.");
}

// Debug: Log session data
error_log("Session Data: " . json_encode($_SESSION));

// Fetch dashboard metrics (cached for 5 minutes, branch-specific)
$cache_key = 'dashboard_metrics_' . md5($user_id . $selected_branch_id . date('Y-m-d H:i'));
$metrics = $_SESSION[$cache_key] ?? null;
$today = date('Y-m-d'); // Define $today globally
if (!$metrics) {
    // Total Invoices (no branch filter)
    $query = "SELECT COUNT(*) AS count, COALESCE(SUM(total_amount), 0) AS total FROM invoices WHERE invoice_date <= ?";
    $params = [$today];
    $types = 's';
    $stmt = $conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed for Invoices Query: $query, Error: " . $conn->error);
        die("Database error. Please try again later.");
    }
    error_log("Invoices Query: $query, Types: $types, Params: " . json_encode($params));
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $invoices = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    // Total Payments (no branch filter)
    $query = "SELECT COUNT(*) AS count, COALESCE(SUM(amount), 0) AS total FROM payments WHERE payment_date <= ?";
    $params = [$today];
    $types = 's';
    $stmt = $conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed for Payments Query: $query, Error: " . $conn->error);
        die("Database error. Please try again later.");
    }
    error_log("Payments Query: $query, Types: $types, Params: " . json_encode($params));
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $payments = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    // Overdue Invoices (global)
    $stmt = $conn->prepare("SELECT COUNT(*) AS count FROM invoices WHERE due_date < ? AND status != 'Paid'");
    if ($stmt === false) {
        error_log("Prepare failed for Overdue Query: Error: " . $conn->error);
        die("Database error. Please try again later.");
    }
    $stmt->bind_param('s', $today);
    $stmt->execute();
    $overdue = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    // Pending Approvals (global)
    $stmt = $conn->prepare("SELECT COUNT(*) AS count FROM invoices WHERE status = 'Pending'");
    if ($stmt === false) {
        error_log("Prepare failed for Pending Query: Error: " . $conn->error);
        die("Database error. Please try again later.");
    }
    $stmt->execute();
    $pending = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    // Petty Cash Pending
    $query = "SELECT COUNT(*) AS count, COALESCE(SUM(amount), 0) AS total FROM petty_cash_transactions WHERE status = 'pending'";
    $params = [];
    $types = '';
    if ($selected_branch_id !== 'all') {
        $query .= " AND branch_id = ?";
        $params[] = $selected_branch_id;
        $types = 'i';
    }
    $stmt = $conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed for Petty Cash Query: $query, Error: " . $conn->error);
        die("Database error. Please try again later.");
    }
    error_log("Petty Cash Query: $query, Types: $types, Params: " . json_encode($params));
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $petty_cash = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    // Credit Notes (global)
    $stmt = $conn->prepare("SELECT COUNT(*) AS count, COALESCE(SUM(amount), 0) AS total FROM credit_notes WHERE credit_date <= ?");
    if ($stmt === false) {
        error_log("Prepare failed for Credit Notes Query: Error: " . $conn->error);
        die("Database error. Please try again later.");
    }
    $stmt->bind_param('s', $today);
    $stmt->execute();
    $credit_notes = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    // Till Reports
    $query = "SELECT COUNT(*) AS count, COALESCE(SUM(d.total_cash), 0) AS total 
              FROM till_report_headers h 
              JOIN (SELECT report_id, SUM(note_1000 + note_500 + note_200 + note_100 + note_50 + coins + drop1 + drop2 + drop3) AS total_cash 
                    FROM till_report_details 
                    GROUP BY report_id) d ON h.id = d.report_id 
              WHERE h.report_date <= ?";
    $params = [$today];
    $types = 's';
    if ($selected_branch_id !== 'all') {
        $query .= " AND h.branch_id = ?";
        $params[] = $selected_branch_id;
        $types .= 'i';
    }
    $stmt = $conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed for Till Reports Query: $query, Error: " . $conn->error);
        die("Database error. Please try again later.");
    }
    error_log("Till Reports Query: $query, Types: $types, Params: " . json_encode($params));
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $till_reports = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    $metrics = [
        'invoices' => ['count' => $invoices['count'], 'total' => $invoices['total']],
        'payments' => ['count' => $payments['count'], 'total' => $payments['total']],
        'overdue' => $overdue['count'],
        'pending' => $pending['count'],
        'petty_cash' => ['count' => $petty_cash['count'], 'total' => $petty_cash['total']],
        'credit_notes' => ['count' => $credit_notes['count'], 'total' => $credit_notes['total']],
        'till_reports' => ['count' => $till_reports['count'], 'total' => $till_reports['total']]
    ];
    $_SESSION[$cache_key] = $metrics;
    $_SESSION[$cache_key . '_expiry'] = time() + 300; // 5 minutes
}

// Fetch notifications
$notifications = [];
if ($metrics['overdue'] > 0) {
    $notifications[] = [
        'type' => 'warning',
        'message' => "{$metrics['overdue']} overdue invoices require immediate action.",
        'link' => '/smartpay360/modules/invoices/list.php?status=Overdue'
    ];
}
if ($metrics['pending'] > 0 && $user_role === 'Admin') {
    $notifications[] = [
        'type' => 'info',
        'message' => "{$metrics['pending']} invoices awaiting your approval.",
        'link' => '/smartpay360/modules/invoices/approve.php'
    ];
}
if ($metrics['petty_cash']['count'] > 0 && $user_role === 'Admin') {
    $notifications[] = [
        'type' => 'warning',
        'message' => "{$metrics['petty_cash']['count']} petty cash transactions pending approval.",
        'link' => '/smartpay360/modules/petty_cash/list.php' . ($selected_branch_id !== 'all' ? "?branch_id=$selected_branch_id" : '')
    ];
}
if ($metrics['credit_notes']['count'] > 0) {
    $notifications[] = [
        'type' => 'info',
        'message' => "{$metrics['credit_notes']['count']} credit notes issued.",
        'link' => '/smartpay360/modules/credit_notes/list.php'
    ];
}
$unbanked_query = "SELECT SUM(d.total_cash) - COALESCE(SUM(b.deposit_amount), 0) AS unbanked 
                   FROM (SELECT report_id, SUM(note_1000 + note_500 + note_200 + note_100 + note_50 + coins + drop1 + drop2 + drop3) AS total_cash 
                         FROM till_report_details GROUP BY report_id) d 
                   JOIN till_report_headers h ON h.id = d.report_id 
                   LEFT JOIN bank_deposits b ON b.report_id = d.report_id 
                   WHERE h.report_date <= ?" . ($selected_branch_id !== 'all' ? " AND h.branch_id = ?" : "");
$params = [$today];
$types = 's';
if ($selected_branch_id !== 'all') {
    $params[] = $selected_branch_id;
    $types .= 'i';
}
$stmt = $conn->prepare($unbanked_query);
if ($stmt === false) {
    error_log("Prepare failed for Unbanked Query: $unbanked_query, Error: " . $conn->error);
    die("Database error. Please try again later.");
}
error_log("Unbanked Query: $unbanked_query, Types: $types, Params: " . json_encode($params));
$stmt->bind_param($types, ...$params);
$stmt->execute();
$unbanked = $stmt->get_result()->fetch_assoc()['unbanked'] ?? 0;
$stmt->close();
if ($unbanked > 0 && $user_role === 'Admin') {
    $notifications[] = [
        'type' => 'danger',
        'message' => "KSh " . number_format($unbanked, 2) . " in till reports remains unbanked.",
        'link' => '/smartpay360/modules/till_reports/list_till_reports.php' . ($selected_branch_id !== 'all' ? "?branch_id=$selected_branch_id" : '')
    ];
}

// Fetch recent transactions (with filter support)
$filter_type = $_GET['filter_type'] ?? 'all';
$filter_date = $_GET['filter_date'] ?? '';
$filter_supplier = $_GET['filter_supplier'] ?? '';

$recent_query = "
    SELECT 
        t.transaction_date, t.description, t.amount, t.transaction_type, t.reference_id, 
        COALESCE(s.name, 'Unknown') AS supplier_name, s.id AS supplier_id
    FROM (
        SELECT invoice_date AS transaction_date, CONCAT('Invoice #', invoice_number) AS description, 
               total_amount AS amount, 'invoice' AS transaction_type, id AS reference_id, supplier_id
        FROM invoices
        UNION ALL
        SELECT payment_date AS transaction_date, CONCAT('Payment - Invoices: ', COALESCE((
            SELECT GROUP_CONCAT(i.invoice_number ORDER BY i.invoice_number SEPARATOR ', ') 
            FROM payment_allocations pa 
            JOIN invoices i ON pa.invoice_id = i.id 
            WHERE pa.payment_id = p.id
        ), 'None')) AS description, amount AS amount, 'payment' AS transaction_type, id AS reference_id, supplier_id
        FROM payments p
    ) t
    LEFT JOIN suppliers s ON t.supplier_id = s.id
    WHERE 1=1
";
$params = [];
$types = '';
if ($filter_type !== 'all') {
    $recent_query .= " AND t.transaction_type = ?";
    $params[] = $filter_type;
    $types .= 's';
}
if ($filter_date) {
    $recent_query .= " AND DATE(t.transaction_date) = ?";
    $params[] = $filter_date;
    $types .= 's';
}
if ($filter_supplier) {
    $recent_query .= " AND s.id = ?";
    $params[] = $filter_supplier;
    $types .= 'i';
}

$recent_query .= " ORDER BY t.transaction_date DESC LIMIT 5";

$stmt = $conn->prepare($recent_query);
if ($stmt === false) {
    error_log("Prepare failed for Recent Transactions Query: $recent_query, Error: " . $conn->error);
    die("Database error. Please try again later.");
}
error_log("Recent Transactions Query: $recent_query, Types: $types, Params: " . json_encode($params));
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$recent_result = $stmt->get_result();
$stmt->close();

// Fetch suppliers for filter dropdown
$suppliers_query = "SELECT id, name FROM suppliers ORDER BY name";
$suppliers_result = $conn->query($suppliers_query);
if ($suppliers_result === false) {
    error_log("Suppliers Query failed: " . $conn->error);
    die("Database error. Please try again later.");
}

// Debug: Log metrics
error_log("Dashboard Metrics: " . json_encode($metrics));
?>

<style>
    .dashboard-card {
        background: #fff;
        border-radius: 8px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        padding: 20px;
        transition: transform 0.2s, box-shadow 0.2s;
        height: 100%;
        position: relative;
        overflow: hidden;
    }
    .dashboard-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    }
    .dashboard-card h5 {
        color: #4b6cb7;
        font-weight: bold;
        margin-bottom: 15px;
    }
    .dashboard-card .metric {
        font-size: 2em;
        color: #343a40;
        font-weight: bold;
    }
    .dashboard-card .sub-metric {
        font-size: 1em;
        color: #6c757d;
    }
    .quick-links .btn {
        border-radius: 6px;
        margin-bottom: 10px;
        width: 100%;
        text-align: left;
        transition: background-color 0.2s;
    }
    .quick-links .btn:hover {
        background-color: #e7f1ff;
    }
    .transactions-table {
        background: #fff;
        border-radius: 8px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        padding: 20px;
    }
    .transactions-table th, .transactions-table td {
        vertical-align: middle;
    }
    .welcome-section {
        background: #e7f1ff;
        border-radius: 8px;
        padding: 20px;
        margin-bottom: 20px;
        animation: fadeIn 0.5s ease-in;
    }
    .notification-card {
        background: #fff;
        border-radius: 8px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        padding: 15px;
        margin-bottom: 20px;
    }
    .notification-card .alert {
        margin-bottom: 0;
        font-size: 0.9em;
    }
    .filter-section {
        margin-bottom: 20px;
    }
    .filter-section .form-select, .filter-section .form-control {
        border-radius: 6px;
        font-size: 0.9em;
    }
    .collapsible-section {
        cursor: pointer;
        padding: 10px;
        background: #f8f9fa;
        border-radius: 6px;
        margin-bottom: 10px;
    }
    .collapsible-section:hover {
        background: #e7f1ff;
    }
    .nav-tabs {
        border-bottom: 2px solid #4b6cb7;
        margin-bottom: 20px;
    }
    .nav-tabs .nav-link {
        border-radius: 6px 6px 0 0;
        color: #4b6cb7;
        font-weight: 500;
        transition: background-color 0.2s, color 0.2s;
    }
    .nav-tabs .nav-link:hover {
        background-color: #e7f1ff;
    }
    .nav-tabs .nav-link.active {
        background-color: #4b6cb7;
        color: #fff;
        border-color: #4b6cb7;
    }
    @keyframes fadeIn {
        from { opacity: 0; transform: translateY(-10px); }
        to { opacity: 1; transform: translateY(0); }
    }
    @media (max-width: 991px) {
        .main-content {
            margin-left: 0 !important;
        }
        .dashboard-card .metric {
            font-size: 1.5em;
        }
        .nav-tabs {
            flex-wrap: nowrap;
            overflow-x: auto;
        }
    }
    @media (max-width: 576px) {
        .quick-links .btn {
            font-size: 0.9em;
        }
        .transactions-table {
            font-size: 0.85em;
        }
        .notification-card .alert {
            font-size: 0.8em;
        }
        .nav-tabs .nav-link {
            font-size: 0.9em;
            padding: 8px;
        }
    }
</style>

<main class="main-content" role="main" aria-label="Dashboard content">
    <!-- Branch Tabs -->
    <ul class="nav nav-tabs" id="branchTabs" role="tablist">
        <li class="nav-item" role="presentation">
            <a class="nav-link <?= $selected_branch_id === 'all' ? 'active' : '' ?>" 
               href="?branch_id=all" 
               id="all-tab" 
               role="tab" 
               aria-controls="all" 
               aria-selected="<?= $selected_branch_id === 'all' ? 'true' : 'false' ?>">
                All Branches
            </a>
        </li>
        <?php while ($branch = $branches_result->fetch_assoc()): ?>
            <?php if ($user_role === 'Admin' || $branch['id'] == $user_branch_id): ?>
                <li class="nav-item" role="presentation">
                    <a class="nav-link <?= $selected_branch_id == $branch['id'] ? 'active' : '' ?>" 
                       href="?branch_id=<?= $branch['id'] ?>" 
                       id="branch-<?= $branch['id'] ?>-tab" 
                       role="tab" 
                       aria-controls="branch-<?= $branch['id'] ?>" 
                       aria-selected="<?= $selected_branch_id == $branch['id'] ? 'true' : 'false' ?>">
                        <?= htmlspecialchars($branch['branch_name']) ?>
                    </a>
                </li>
            <?php endif; ?>
        <?php endwhile; ?>
        <?php $branches_result->data_seek(0); // Reset for later use ?>
    </ul>

    <!-- Welcome Section -->
    <div class="welcome-section">
        <h2 class="fw-bold text-primary" tabindex="0">Welcome, <?= htmlspecialchars($user_name) ?>!</h2>
        <p class="text-muted">
            <?php if ($user_role === 'Admin'): ?>
                Manage all operations and oversee the financial health of SmartPay360.
            <?php else: ?>
                Monitor your transactions and streamline payments efficiently.
            <?php endif; ?>
            <?php if ($selected_branch_id !== 'all'): ?>
                <br>Currently viewing: 
                <?php 
                    $stmt = $conn->prepare("SELECT branch_name FROM branches WHERE id = ?");
                    if ($stmt === false) {
                        error_log("Prepare failed for Branch Name Query: Error: " . $conn->error);
                        echo "Unknown Branch";
                    } else {
                        $stmt->bind_param('i', $selected_branch_id);
                        $stmt->execute();
                        $branch_name = $stmt->get_result()->fetch_assoc()['branch_name'] ?? 'Unknown';
                        $stmt->close();
                        echo htmlspecialchars($branch_name);
                    }
                ?>
            <?php endif; ?>
        </p>
    </div>

    <!-- Notifications -->
    <?php if (!empty($notifications)): ?>
        <div class="notification-card" role="alert" aria-live="polite">
            <h5 class="text-dark"><i class="fas fa-bell me-2"></i>Notifications</h5>
            <?php foreach ($notifications as $notification): ?>
                <div class="alert alert-<?= $notification['type'] ?> d-flex align-items-center" role="alert">
                    <i class="fas fa-<?= $notification['type'] === 'warning' ? 'exclamation-circle' : ($notification['type'] === 'danger' ? 'exclamation-triangle' : 'info-circle') ?> me-2"></i>
                    <div>
                        <?= htmlspecialchars($notification['message']) ?>
                        <a href="<?= htmlspecialchars($notification['link']) ?>" class="alert-link">View</a>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

    <!-- Financial Metrics (Collapsible) -->
    <div class="collapsible-section" data-bs-toggle="collapse" data-bs-target="#metricsCollapse" aria-expanded="true" aria-controls="metricsCollapse" role="button" tabindex="0">
        <h5 class="mb-0"><i class="fas fa-chart-bar me-2"></i>Financial Metrics</h5>
    </div>
    <div class="collapse show" id="metricsCollapse">
        <div class="row g-4 mb-4">
            <div class="col-lg-3 col-md-6">
                <div class="dashboard-card" role="region" aria-label="Total Invoices">
                    <h5><i class="fas fa-file-invoice me-2 text-danger"></i>Total Invoices</h5>
                    <div class="metric" tabindex="0"><?= number_format($metrics['invoices']['count']) ?></div>
                    <div class="sub-metric">KSh <?= number_format($metrics['invoices']['total'], 2) ?></div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="dashboard-card" role="region" aria-label="Total Payments">
                    <h5><i class="fas fa-money-check me-2 text-success"></i>Total Payments</h5>
                    <div class="metric" tabindex="0"><?= number_format($metrics['payments']['count']) ?></div>
                    <div class="sub-metric">KSh <?= number_format($metrics['payments']['total'], 2) ?></div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="dashboard-card" role="region" aria-label="Overdue Invoices">
                    <h5><i class="fas fa-exclamation-circle me-2 text-warning"></i>Overdue Invoices</h5>
                    <div class="metric" tabindex="0"><?= number_format($metrics['overdue']) ?></div>
                    <div class="sub-metric">Action Required</div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="dashboard-card" role="region" aria-label="Pending Approvals">
                    <h5><i class="fas fa-hourglass-half me-2 text-info"></i>Pending Approvals</h5>
                    <div class="metric" tabindex="0"><?= number_format($metrics['pending']) ?></div>
                    <div class="sub-metric">Review Now</div>
                </div>
            </div>
        </div>
    </div>

    <!-- Additional Metrics (Petty Cash, Credit Notes, Till Reports) -->
    <div class="collapsible-section" data-bs-toggle="collapse" data-bs-target="#additionalMetricsCollapse" aria-expanded="true" aria-controls="additionalMetricsCollapse" role="button" tabindex="0">
        <h5 class="mb-0"><i class="fas fa-wallet me-2"></i>Additional Metrics</h5>
    </div>
    <div class="collapse show" id="additionalMetricsCollapse">
        <div class="row g-4 mb-4">
            <div class="col-lg-4 col-md-6">
                <div class="dashboard-card" role="region" aria-label="Pending Petty Cash">
                    <h5><i class="fas fa-money-bill me-2 text-warning"></i>Pending Petty Cash</h5>
                    <div class="metric" tabindex="0"><?= number_format($metrics['petty_cash']['count']) ?></div>
                    <div class="sub-metric">KSh <?= number_format($metrics['petty_cash']['total'], 2) ?></div>
                </div>
            </div>
            <div class="col-lg-4 col-md-6">
                <div class="dashboard-card" role="region" aria-label="Credit Notes">
                    <h5><i class="fas fa-clipboard-list me-2 text-secondary"></i>Credit Notes</h5>
                    <div class="metric" tabindex="0"><?= number_format($metrics['credit_notes']['count']) ?></div>
                    <div class="sub-metric">KSh <?= number_format($metrics['credit_notes']['total'], 2) ?></div>
                </div>
            </div>
            <div class="col-lg-4 col-md-6">
                <div class="dashboard-card" role="region" aria-label="Till Reports">
                    <h5><i class="fas fa-cash-register me-2 text-primary"></i>Till Reports</h5>
                    <div class="metric" tabindex="0"><?= number_format($metrics['till_reports']['count']) ?></div>
                    <div class="sub-metric">KSh <?= number_format($metrics['till_reports']['total'], 2) ?></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Quick Links -->
    <div class="collapsible-section" data-bs-toggle="collapse" data-bs-target="#quickLinksCollapse" aria-expanded="true" aria-controls="quickLinksCollapse" role="button" tabindex="0">
        <h5 class="mb-0"><i class="fas fa-link me-2"></i>Quick Actions</h5>
    </div>
    <div class="collapse show" id="quickLinksCollapse">
        <div class="row g-4 mb-4">
            <div class="col-lg-3 col-md-12">
                <div class="dashboard-card quick-links">
                    <a href="/smartpay360/modules/invoices/add.php" class="btn btn-outline-primary"><i class="fas fa-file-invoice me-2"></i>Add Invoice</a>
                    <a href="/smartpay360/modules/payments/add.php" class="btn btn-outline-success"><i class="fas fa-money-check me-2"></i>Add Payment</a>
                    <a href="/smartpay360/modules/petty_cash/add.php<?= $selected_branch_id !== 'all' ? '?branch_id=' . $selected_branch_id : '' ?>" class="btn btn-outline-warning"><i class="fas fa-money-bill me-2"></i>Add Petty Cash</a>
                    <a href="/smartpay360/modules/credit_notes/add.php" class="btn btn-outline-secondary"><i class="fas fa-clipboard-list me-2"></i>Add Credit Note</a>
                    <a href="/smartpay360/modules/payments/calendar.php" class="btn btn-outline-info"><i class="fas fa-calendar-alt me-2"></i>Payments Calendar</a>
                    <a href="/smartpay360/modules/reports/statement.php" class="btn btn-outline-secondary"><i class="fas fa-list-alt me-2"></i>Supplier Statement</a>
                    <a href="/smartpay360/modules/till_reports/list_till_reports.php<?= $selected_branch_id !== 'all' ? '?branch_id=' . $selected_branch_id : '' ?>" class="btn btn-outline-primary"><i class="fas fa-cash-register me-2"></i>Till Reports</a>
                </div>
            </div>
            <div class="col-lg-9 col-md-12">
                <div class="dashboard-card">
                    <h5><i class="fas fa-bolt me-2"></i>Quick Action Center</h5>
                    <p class="text-muted">Perform common tasks quickly.</p>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#quickActionModal" aria-label="Open quick action modal"><i class="fas fa-plus me-2"></i>New Action</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Recent Transactions with Filters -->
    <div class="collapsible-section" data-bs-toggle="collapse" data-bs-target="#transactionsCollapse" aria-expanded="true" aria-controls="transactionsCollapse" role="button" tabindex="0">
        <h5 class="mb-0"><i class="fas fa-history me-2"></i>Recent Transactions</h5>
    </div>
    <div class="collapse show" id="transactionsCollapse">
        <div class="transactions-table">
            <div class="filter-section">
                <form method="GET" class="row g-3" role="search" aria-label="Transaction filters">
                    <input type="hidden" name="branch_id" value="<?= htmlspecialchars($selected_branch_id) ?>">
                    <div class="col-md-3">
                        <label for="filter_type" class="visually-hidden">Transaction Type</label>
                        <select name="filter_type" id="filter_type" class="form-select">
                            <option value="all" <?= $filter_type === 'all' ? 'selected' : '' ?>>All Types</option>
                            <option value="invoice" <?= $filter_type === 'invoice' ? 'selected' : '' ?>>Invoices</option>
                            <option value="payment" <?= $filter_type === 'payment' ? 'selected' : '' ?>>Payments</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="filter_date" class="visually-hidden">Date</label>
                        <input type="date" name="filter_date" id="filter_date" class="form-control" value="<?= htmlspecialchars($filter_date) ?>">
                    </div>
                    <div class="col-md-3">
                        <label for="filter_supplier" class="visually-hidden">Supplier</label>
                        <select name="filter_supplier" id="filter_supplier" class="form-select">
                            <option value="">All Suppliers</option>
                            <?php while ($supplier = $suppliers_result->fetch_assoc()): ?>
                                <option value="<?= $supplier['id'] ?>" <?= $filter_supplier == $supplier['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($supplier['name']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <button type="submit" class="btn btn-primary w-100"><i class="fas fa-filter me-2"></i>Filter</button>
                    </div>
                </form>
            </div>
            <table class="table table-sm table-bordered" aria-label="Recent transactions">
                <thead>
                    <tr>
                        <th scope="col">Date</th>
                        <th scope="col">Supplier</th>
                        <th scope="col">Description</th>
                        <th scope="col">Amount</th>
                        <th scope="col">Type</th>
                        <th scope="col">Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($recent_result->num_rows > 0): ?>
                        <?php while ($row = $recent_result->fetch_assoc()): ?>
                            <tr>
                                <td><?= date('d-M-Y', strtotime($row['transaction_date'])) ?></td>
                                <td><?= htmlspecialchars($row['supplier_name']) ?></td>
                                <td><?= htmlspecialchars($row['description']) ?></td>
                                <td class="text-end">KSh <?= number_format($row['amount'], 2) ?></td>
                                <td>
                                    <span class="badge <?= $row['transaction_type'] === 'invoice' ? 'bg-danger' : 'bg-success' ?>">
                                        <?= ucfirst($row['transaction_type']) ?>
                                    </span>
                                </td>
                                <td>
                                    <a href="/smartpay360/modules/<?= $row['transaction_type'] === 'invoice' ? 'invoices' : 'payments' ?>/view.php?id=<?= $row['reference_id'] ?>" 
                                       class="btn btn-sm btn-outline-primary" 
                                       aria-label="View <?= $row['transaction_type'] ?> details">
                                        View
                                    </a>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="6" class="text-center">No transactions found.</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Quick Action Modal -->
    <div class="modal fade" id="quickActionModal" tabindex="-1" aria-labelledby="quickActionModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="quickActionModalLabel">Quick Action</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <p>Select an action to perform:</p>
                    <div class="d-grid gap-2">
                        <a href="/smartpay360/modules/invoices/add.php" class="btn btn-primary">Add New Invoice</a>
                        <a href="/smartpay360/modules/payments/add.php" class="btn btn-success">Record Payment</a>
                        <a href="/smartpay360/modules/petty_cash/add.php<?= $selected_branch_id !== 'all' ? '?branch_id=' . $selected_branch_id : '' ?>" class="btn btn-warning">Add Petty Cash</a>
                        <a href="/smartpay360/modules/credit_notes/add.php" class="btn btn-secondary">Add Credit Note</a>
                        <a href="/smartpay360/modules/till_reports/list_till_reports.php<?= $selected_branch_id !== 'all' ? '?branch_id=' . $selected_branch_id : '' ?>" class="btn btn-primary">View Till Reports</a>
                        <a href="/smartpay360/modules/invoices/approve.php" class="btn btn-info">Review Pending Approvals</a>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
</main>

<script>
// Collapsible section accessibility
document.querySelectorAll('.collapsible-section').forEach(section => {
    section.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
            e.preventDefault();
            section.click();
        }
    });
});

// Notification fade-in
document.querySelectorAll('.notification-card').forEach(card => {
    card.style.opacity = 0;
    setTimeout(() => {
        card.style.transition = 'opacity 0.5s ease-in';
        card.style.opacity = 1;
    }, 100);
});

// Tab accessibility
document.querySelectorAll('.nav-tabs .nav-link').forEach(tab => {
    tab.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
            e.preventDefault();
            tab.click();
        }
    });
});
</script>

<?php
// Clean up expired cache
if (isset($_SESSION[$cache_key . '_expiry']) && $_SESSION[$cache_key . '_expiry'] < time()) {
    unset($_SESSION[$cache_key], $_SESSION[$cache_key . '_expiry']);
}
?>