<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

// Initialize filters
$supplier_id = isset($_GET['supplier_id']) ? (int)$_GET['supplier_id'] : 0;
$from_date = $_GET['from_date'] ?? '';
$to_date = $_GET['to_date'] ?? '';

// Build query conditions
$conditions = [];
$params = [];
$types = '';

if ($supplier_id > 0) {
    $conditions[] = "p.supplier_id = ?";
    $params[] = $supplier_id;
    $types .= 'i';
}
if (!empty($from_date)) {
    $conditions[] = "p.payment_date >= ?";
    $params[] = $from_date;
    $types .= 's';
}
if (!empty($to_date)) {
    $conditions[] = "p.payment_date <= ?";
    $params[] = $to_date;
    $types .= 's';
}
$where = count($conditions) ? 'WHERE ' . implode(' AND ', $conditions) : '';

// Query for payment events
$query = "
    SELECT p.id, p.supplier_id, s.name AS supplier_name, p.amount, p.payment_date,
           (SELECT GROUP_CONCAT(i.invoice_number ORDER BY i.invoice_number SEPARATOR ', ') 
            FROM payment_allocations pa 
            JOIN invoices i ON pa.invoice_id = i.id 
            WHERE pa.payment_id = p.id) AS invoice_numbers
    FROM payments p
    JOIN suppliers s ON p.supplier_id = s.id
    $where
    ORDER BY p.payment_date ASC
";

$stmt = $conn->prepare($query);
if (count($params) > 0) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();

// Prepare events for FullCalendar
$events = [];
while ($row = $result->fetch_assoc()) {
    $events[] = [
        'id' => $row['id'],
        'title' => htmlspecialchars($row['supplier_name'] . ' - KSh ' . number_format($row['amount'], 2)),
        'start' => $row['payment_date'],
        'extendedProps' => [
            'supplier' => htmlspecialchars($row['supplier_name']),
            'amount' => 'KSh ' . number_format($row['amount'], 2),
            'invoices' => htmlspecialchars($row['invoice_numbers'] ?: 'None'),
            'payment_id' => $row['id']
        ]
    ];
}
$events_json = json_encode($events);

// Supplier dropdown
$suppliers = $conn->query("SELECT id, name FROM suppliers ORDER BY name ASC");

// Total summary
$total_query = "
    SELECT SUM(p.amount) AS total_paid
    FROM payments p
    $where
";
$total_stmt = $conn->prepare($total_query);
if (count($params) > 0) {
    $total_stmt->bind_param($types, ...$params);
}
$total_stmt->execute();
$total_result = $total_stmt->get_result()->fetch_assoc();
$total_paid = $total_result['total_paid'] ?? 0;
?>

<div class="main-content">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h4 class="fw-bold text-primary"><i class="fas fa-calendar-alt me-2"></i> Supplier Payments Calendar</h4>
        <div>
            <a href="add.php" class="btn btn-primary me-2"><i class="fas fa-plus-circle me-1"></i> Add Payment</a>
            <a href="export_csv.php?<?= http_build_query($_GET) ?>" class="btn btn-outline-success btn-sm me-1"><i class="fas fa-file-csv"></i> Export CSV</a>
            <a href="export_pdf.php?<?= http_build_query($_GET) ?>" class="btn btn-outline-danger btn-sm"><i class="fas fa-file-pdf"></i> Export PDF</a>
        </div>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= $_SESSION['success']; unset($_SESSION['success']); ?></div>
    <?php elseif (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error']; unset($_SESSION['error']); ?></div>
    <?php endif; ?>

    <!-- Filter Form -->
    <div class="card shadow-sm mb-3">
        <div class="card-body">
            <form method="GET" class="row g-2 mb-0">
                <div class="col-md-3">
                    <select name="supplier_id" class="form-select form-select-sm">
                        <option value="">-- All Suppliers --</option>
                        <?php while ($supplier = $suppliers->fetch_assoc()): ?>
                            <option value="<?= $supplier['id'] ?>" <?= ($supplier_id == $supplier['id']) ? 'selected' : '' ?>>
                                <?= htmlspecialchars($supplier['name']) ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <input type="date" name="from_date" value="<?= htmlspecialchars($from_date) ?>" class="form-control form-control-sm" placeholder="From Date">
                </div>
                <div class="col-md-2">
                    <input type="date" name="to_date" value="<?= htmlspecialchars($to_date) ?>" class="form-control form-control-sm" placeholder="To Date">
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-sm btn-primary w-100">🔎 Filter</button>
                </div>
                <div class="col-md-2">
                    <a href="calendar.php" class="btn btn-sm btn-secondary w-100">🔄 Reset</a>
                </div>
            </form>
        </div>
    </div>

    <!-- Totals -->
    <div class="alert alert-info fw-bold">
        💰 Total Amount Paid: <span class="text-success">KSh <?= number_format($total_paid, 2) ?></span>
    </div>

    <!-- Calendar Section -->
    <div class="card shadow-sm">
        <div class="card-body">
            <div id="calendar"></div>
        </div>
    </div>
</div>

<!-- FullCalendar and Tippy.js -->
<script src="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.15/index.global.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.11.8/dist/umd/popper.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/tippy.js@6.3.7/dist/tippy-bundle.umd.min.js"></script>
<link href="https://cdn.jsdelivr.net/npm/tippy.js@6.3.7/dist/tippy.css" rel="stylesheet">

<script>
document.addEventListener('DOMContentLoaded', function() {
    var calendarEl = document.getElementById('calendar');
    var calendar = new FullCalendar.Calendar(calendarEl, {
        initialView: 'dayGridMonth',
        headerToolbar: {
            left: 'prev,next today',
            center: 'title',
            right: 'dayGridMonth,timeGridWeek,timeGridDay'
        },
        events: <?= $events_json ?>,
        eventDidMount: function(info) {
            // Add tooltip with payment details using Tippy.js
            tippy(info.el, {
                content: `
                    <strong>Supplier:</strong> ${info.event.extendedProps.supplier}<br>
                    <strong>Amount:</strong> ${info.event.extendedProps.amount}<br>
                    <strong>Invoices:</strong> ${info.event.extendedProps.invoices}<br>
                    <a href="view.php?id=${info.event.extendedProps.payment_id}" class="btn btn-sm btn-outline-primary mt-2">View</a>
                    <a href="receipt.php?id=${info.event.extendedProps.payment_id}" class="btn btn-sm btn-outline-secondary mt-2">PDF</a>
                `,
                allowHTML: true,
                interactive: true,
                theme: 'light-border'
            });
        },
        eventClick: function(info) {
            // Prevent default to handle via tooltip links
            info.jsEvent.preventDefault();
        },
        eventTimeFormat: {
            hour: 'numeric',
            minute: '2-digit',
            meridiem: 'short'
        },
        height: 'auto'
    });
    calendar.render();
});
</script>

<style>
.fc-event {
    cursor: pointer;
    background-color: #007bff !important;
    border-color: #0056b3 !important;
    color: white !important;
    font-size: 13px !important;
}
.fc-event:hover {
    background-color: #0056b3 !important;
}
.tippy-content {
    font-size: 13px;
    padding: 10px;
}
</style>