<?php
session_start();
require_once '../../config/db.php';

$payment_id = (int) ($_GET['id'] ?? 0);
if ($payment_id <= 0) {
    die("Invalid Payment ID.");
}

// Fetch payment + loading details
$stmt = $conn->prepare("
    SELECT bp.*, l.loading_date, l.mileage_start, v.plate_number AS vehicle_no, v.driver_name, r.name AS route_name
    FROM bread_payments bp
    JOIN loadings l ON bp.loading_id = l.id
    JOIN vehicles v ON l.vehicle_id = v.id
    JOIN routes r ON l.route_id = r.id
    WHERE bp.id = ?
");
$stmt->bind_param('i', $payment_id);
$stmt->execute();
$payment = $stmt->get_result()->fetch_assoc();
if (!$payment) {
    die("Payment not found.");
}

// Fetch bread items loaded and returned
$stmt = $conn->prepare("
    SELECT s.name AS sku_name, li.crates_loaded, s.price_per_crate, s.loaves_per_crate,
        IFNULL(SUM(r.returned_crates), 0) AS returned_crates, 
        IFNULL(SUM(r.returned_pieces), 0) AS returned_pieces
    FROM loading_items li
    JOIN skus s ON li.sku_id = s.id
    LEFT JOIN returns r ON li.loading_id = r.loading_id AND li.sku_id = r.sku_id
    WHERE li.loading_id = ?
    GROUP BY li.sku_id
");
$stmt->bind_param('i', $payment['loading_id']);
$stmt->execute();
$items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Calculate totals
$total_expected = 0;
foreach ($items as $item) {
    $loaded = $item['crates_loaded'] * $item['loaves_per_crate'];
    $returned = ($item['returned_crates'] * $item['loaves_per_crate']) + $item['returned_pieces'];
    $sold = $loaded - $returned;
    $sold_crates = $sold / $item['loaves_per_crate'];
    $total_expected += $sold_crates * $item['price_per_crate'];
}

$total_collected = $payment['cash_amount'] + $payment['mpesa_amount'] + $payment['cheque_amount'];
$difference = $total_collected - $total_expected;
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Receipt #<?= $payment_id ?></title>
<style>
    @page { size: 80mm auto; margin: 0; }
    body { width: 80mm; font-family: Arial, sans-serif; font-size: 14px; margin: 0; padding: 5px; color: #000; }
    .center { text-align: center; }
    .bold { font-weight: bold; }
    .line { border-top: 2px dashed #000; margin: 10px 0; }
    table { width: 100%; border-collapse: collapse; font-size: 13px; }
    td { padding: 3px 0; }
    .summary { margin-top: 10px; }
    .signature-area { margin-top: 20px; }
    .small-text { font-size: 11px; text-align: center; margin-top: 15px; }
    @media print { body { zoom: 1.7; } }
</style>
<script src="https://cdn.jsdelivr.net/npm/qrious/dist/qrious.min.js"></script>
</head>

<body onload="window.print()">
<div class="container">

<!-- Header -->
<div class="center">
    <h3>DELTA SUPERSTORES LTD</h3>
    <div class="line"></div>
    <p class="bold">PAYMENT RECEIPT</p>
</div>

<!-- Details -->
<p>
<strong>Date:</strong> <?= htmlspecialchars($payment['payment_date']) ?> <?= date('H:i') ?><br>
<strong>Vehicle:</strong> <?= htmlspecialchars($payment['vehicle_no']) ?><br>
<strong>Driver:</strong> <?= htmlspecialchars($payment['driver_name']) ?><br>
<strong>Route:</strong> <?= htmlspecialchars($payment['route_name']) ?><br>
<strong>Mileage Start:</strong> <?= htmlspecialchars($payment['mileage_start']) ?><br>
</p>

<div class="line"></div>

<!-- Item Breakdown -->
<table>
    <thead>
        <tr><td class="bold" colspan="2">Bread Sales</td></tr>
    </thead>
    <tbody>
    <?php foreach ($items as $item):
        $loaded = $item['crates_loaded'] * $item['loaves_per_crate'];
        $returned = ($item['returned_crates'] * $item['loaves_per_crate']) + $item['returned_pieces'];
        $sold = $loaded - $returned;
        $sold_crates_equivalent = $sold / $item['loaves_per_crate'];
        $value = $sold_crates_equivalent * $item['price_per_crate'];
    ?>
    <tr>
        <td><?= htmlspecialchars($item['sku_name']) ?> x <?= $sold ?> pcs</td>
        <td style="text-align:right;">Ksh <?= number_format($value, 2) ?></td>
    </tr>
    <?php endforeach; ?>
    </tbody>
</table>

<div class="line"></div>

<!-- Payments -->
<p><strong>Payment Breakdown:</strong></p>
<table>
    <?php if ($payment['cash_amount'] > 0): ?>
    <tr><td>Cash:</td><td style="text-align:right;">Ksh <?= number_format($payment['cash_amount'], 2) ?></td></tr>
    <?php endif; ?>
    <?php if ($payment['mpesa_amount'] > 0): ?>
    <tr><td>MPESA:</td><td style="text-align:right;">Ksh <?= number_format($payment['mpesa_amount'], 2) ?></td></tr>
    <?php endif; ?>
    <?php if ($payment['cheque_amount'] > 0): ?>
    <tr><td>Cheque:</td><td style="text-align:right;">Ksh <?= number_format($payment['cheque_amount'], 2) ?></td></tr>
    <?php endif; ?>
</table>

<div class="line"></div>

<!-- Totals -->
<div class="summary">
    <p><strong>Total Expected:</strong> Ksh <?= number_format($total_expected, 2) ?></p>
    <p><strong>Total Collected:</strong> Ksh <?= number_format($total_collected, 2) ?></p>
    <p><strong>Difference:</strong> 
        <?php if ($difference > 0): ?>
            <span style="color:green;">Overpaid Ksh <?= number_format($difference, 2) ?></span>
        <?php elseif ($difference < 0): ?>
            <span style="color:red;">Underpaid Ksh <?= number_format(abs($difference), 2) ?></span>
        <?php else: ?>
            Exact (0.00)
        <?php endif; ?>
    </p>
</div>

<div class="line"></div>

<!-- Signatures -->
<div class="signature-area">___________________________<br>Sales Person Signature</div>
<div class="signature-area">___________________________<br>Cashier Signature</div>

<!-- Footer -->
<div class="small-text">Printed by SmartPay360</div>

<!-- QR Code -->
<div class="center" style="margin-top:10px;">
    <canvas id="qr-code"></canvas>
</div>

<script>
var qr = new QRious({
    element: document.getElementById('qr-code'),
    value: "PAYMENT-ID:<?= $payment_id ?>",
    size: 80,
    level: 'H'
});
</script>

</div>
</body>
</html>
