<?php
require_once 'pdo_db.php';
session_start();

// Enable error logging
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', 'C:/xampp/php/logs/php_error_log');

header('Content-Type: application/json');

// CSRF validation
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    error_log('CSRF token mismatch: Received ' . ($_POST['csrf_token'] ?? 'none') . ', Expected ' . ($_SESSION['csrf_token'] ?? 'none'));
    echo json_encode(['success' => false, 'error' => 'Invalid CSRF token']);
    exit;
}

// Validate input
$layout_id = isset($_POST['layout_id']) ? (int)$_POST['layout_id'] : 0;
if ($layout_id <= 0 || !isset($_FILES['background'])) {
    error_log('Invalid input: layout_id=' . $layout_id . ', files=' . json_encode($_FILES));
    echo json_encode(['success' => false, 'error' => 'Invalid layout ID or no file uploaded']);
    exit;
}

$file = $_FILES['background'];
$allowed_types = ['image/png', 'image/jpeg'];
$max_size = 1048576; // 1MB

// Validate file
if ($file['error'] !== UPLOAD_ERR_OK) {
    $errors = [
        UPLOAD_ERR_INI_SIZE => 'File exceeds PHP upload size limit',
        UPLOAD_ERR_FORM_SIZE => 'File exceeds form size limit',
        UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
        UPLOAD_ERR_NO_FILE => 'No file was uploaded',
        UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary directory',
        UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
        UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the upload'
    ];
    $error_msg = $errors[$file['error']] ?? 'Unknown upload error';
    error_log('File upload error: ' . $error_msg);
    echo json_encode(['success' => false, 'error' => $error_msg]);
    exit;
}

if (!in_array($file['type'], $allowed_types)) {
    error_log('Invalid file type: ' . $file['type']);
    echo json_encode(['success' => false, 'error' => 'Only PNG and JPEG images are allowed']);
    exit;
}
if ($file['size'] > $max_size) {
    error_log('File too large: ' . $file['size'] . ' bytes');
    echo json_encode(['success' => false, 'error' => 'Image size must be less than 1MB']);
    exit;
}

// Generate unique filename
$ext = pathinfo($file['name'], PATHINFO_EXTENSION);
$filename = 'cheque_' . $layout_id . '_' . time() . '.' . $ext;
$upload_dir = '../../Uploads/cheque_backgrounds/';
$upload_path = $upload_dir . $filename;

// Ensure upload directory exists
if (!is_dir($upload_dir)) {
    if (!mkdir($upload_dir, 0755, true)) {
        error_log('Failed to create upload directory: ' . $upload_dir);
        echo json_encode(['success' => false, 'error' => 'Failed to create upload directory']);
        exit;
    }
}

// Check directory permissions
if (!is_writable($upload_dir)) {
    error_log('Upload directory not writable: ' . $upload_dir);
    echo json_encode(['success' => false, 'error' => 'Upload directory is not writable']);
    exit;
}

try {
    // Move uploaded file
    if (!move_uploaded_file($file['tmp_name'], $upload_path)) {
        error_log('Failed to move uploaded file to: ' . $upload_path);
        throw new Exception('Failed to save image');
    }

    // Verify file exists and is readable
    if (!file_exists($upload_path) || !is_readable($upload_path)) {
        error_log('Uploaded file not found or not readable: ' . $upload_path);
        throw new Exception('Uploaded file is inaccessible');
    }

    // Update database
    $stmt = $pdo->prepare("UPDATE cheque_layouts SET background_image = ? WHERE id = ?");
    if (!$stmt->execute([$filename, $layout_id])) {
        error_log('Database update failed for layout_id: ' . $layout_id);
        throw new Exception('Failed to update database');
    }

    error_log('Image uploaded successfully for layout_id: ' . $layout_id);
    echo json_encode(['success' => true, 'filename' => $filename]);
} catch (Exception $e) {
    error_log('Upload error: ' . $e->getMessage());
    echo json_encode(['success' => false, 'error' => 'Upload error: ' . $e->getMessage()]);
}
?>