<?php
session_start();
require_once '../../config/db.php';

// Validate input
if (!isset($_POST['selected_invoice_json'])) {
  $_SESSION['error'] = "No invoice selection found.";
  header("Location: select_invoices.php");
  exit;
}

$invoice_allocations = json_decode($_POST['selected_invoice_json'], true);
if (!is_array($invoice_allocations) || count($invoice_allocations) === 0) {
  $_SESSION['error'] = "Invalid invoice selection.";
  header("Location: select_invoices.php");
  exit;
}

// Validate supplier consistency
$supplier_ids = array_unique(array_column($invoice_allocations, 'supplier_id'));
if (count($supplier_ids) > 1) {
  $_SESSION['error'] = "All selected invoices must belong to the same supplier.";
  header("Location: select_invoices.php");
  exit;
}
$supplier_id = $supplier_ids[0] ?? null;

// Generate CSRF token
$_SESSION['csrf_token'] = bin2hex(random_bytes(32));

require_once '../../includes/layout2.php';

// Fetch invoice details
$ids = array_column($invoice_allocations, 'invoice_id');
$placeholders = implode(',', array_fill(0, count($ids), '?'));
$query = "
  SELECT 
  i.id, 
  i.invoice_number, 
  i.supplier_id, 
  s.name AS supplier_name, 
  b.branch_name,
  i.total_payable,
  IFNULL(paid.total_paid, 0) AS paid,
  IFNULL(credit.total_credit, 0) AS credit
FROM invoices i
JOIN suppliers s ON i.supplier_id = s.id
JOIN branches b ON i.branch_id = b.id
LEFT JOIN (
  SELECT invoice_id, SUM(amount_allocated) AS total_paid
  FROM payment_allocations
  GROUP BY invoice_id
) paid ON paid.invoice_id = i.id
LEFT JOIN (
  SELECT invoice_id, SUM(amount) AS total_credit
  FROM credit_notes
  GROUP BY invoice_id
) credit ON credit.invoice_id = i.id
WHERE i.id IN ($placeholders)
";

$stmt = $conn->prepare($query);
if (!$stmt) {
  $_SESSION['error'] = "Database error: " . $conn->error;
  header("Location: select_invoices.php");
  exit;
}
$stmt->bind_param(str_repeat('i', count($ids)), ...$ids);
$stmt->execute();
$result = $stmt->get_result();

$invoices = [];
$totalInvoice = 0;
while ($row = $result->fetch_assoc()) {
  $row['balance'] = $row['total_payable'] - $row['paid'] - $row['credit'];
  $totalInvoice += $row['balance'];
  $invoices[$row['id']] = $row;
}

if (count($invoices) !== count($ids)) {
  $_SESSION['error'] = "Some selected invoices are invalid or not found.";
  header("Location: select_invoices.php");
  exit;
}
?>

<div class="main-content">
  <h4 class="fw-bold mb-4 text-primary"><i class="fas fa-money-check-alt me-2"></i>Confirm Payment & Allocation</h4>

  <form method="POST" action="save_payment.php" id="paymentForm">
    <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($_SESSION['csrf_token']) ?>">
    <input type="hidden" name="invoice_allocations_json" id="invoice_allocations_json">
    <input type="hidden" name="supplier_id" value="<?= htmlspecialchars($supplier_id) ?>">

    <button class="btn btn-outline-dark mb-2" type="button" data-bs-toggle="collapse" data-bs-target="#allocSection">
      🧾 Toggle Invoice Allocation (Total: KSh <?= number_format($totalInvoice, 2) ?>)
    </button>

    <div class="collapse show" id="allocSection">
      <table class="table table-bordered align-middle">
        <thead class="table-light">
          <tr>
            <th>Invoice No</th>
            <th>Supplier</th>
            <th>Branch</th>
            <th>Total</th>
            <th>Paid</th>
            <th>Credit</th>
            <th>Balance</th>
            <th>Allocate</th>
          </tr>
        </thead>
        <tbody>
        <?php foreach ($invoice_allocations as $item): 
          $inv = $invoices[$item['invoice_id']];
        ?>
          <tr>
            <td><?= htmlspecialchars($inv['invoice_number']) ?></td>
            <td><?= htmlspecialchars($inv['supplier_name']) ?></td>
            <td><?= htmlspecialchars($inv['branch_name']) ?></td>
            <td class="text-end"><?= number_format($inv['total_payable'], 2) ?></td>
            <td class="text-end text-muted"><?= number_format($inv['paid'], 2) ?></td>
            <td class="text-end text-warning"><?= number_format($inv['credit'], 2) ?></td>
            <td class="text-end text-success fw-bold"><?= number_format($inv['balance'], 2) ?></td>
            <td>
              <input type="number" step="0.01" min="0" max="<?= $inv['balance'] ?>" 
                     class="form-control form-control-sm alloc" data-id="<?= $inv['id'] ?>" 
                     name="allocations[<?= $inv['id'] ?>]"
                     value="<?= isset($item['amount']) ? number_format($item['amount'], 2, '.', '') : '' ?>">
            </td>
          </tr>
        <?php endforeach; ?>
        </tbody>
      </table>
      <div id="allocatedSummary" class="fw-bold text-end mb-3">
        Allocated Total: <span id="allocatedTotal">KSh 0.00</span>
      </div>
    </div>

    <!-- Payment Details -->
    <h5 class="mt-4 text-primary"><i class="fas fa-credit-card me-2"></i>Payment Details</h5>
    <div class="row g-3">
      <div class="col-md-4">
        <label class="form-label">Payment Method</label>
        <select name="payment_method" class="form-select" required>
          <option value="Cash">Cash</option>
          <option value="Cheque">Cheque</option>
          <option value="MPESA">MPESA</option>
        </select>
      </div>
      <div class="col-md-4">
        <label class="form-label">Reference No</label>
        <input type="text" name="reference_no" class="form-control" required placeholder="Cheque No / Txn ID">
      </div>
      <div class="col-md-4">
        <label class="form-label">Payment Date</label>
        <input type="date" name="payment_date" class="form-control" value="<?= date('Y-m-d') ?>" required>
      </div>
      <div class="col-md-4">
        <label class="form-label">Amount</label>
        <input type="number" step="0.01" min="0" name="payment_amount" class="form-control amount" required>
      </div>
      <div class="col-md-8">
        <label class="form-label">Remarks</label>
        <input type="text" name="remarks" class="form-control" placeholder="Optional remarks">
      </div>
    </div>

    <div class="alert alert-info fw-bold text-end mt-3">Total Payment: <span id="totalPaid">KSh 0.00</span></div>
    <div class="alert alert-warning fw-bold text-end" id="overpaymentSection" style="display: none;">
      Overpayment: <span id="overpaymentAmount">KSh 0.00</span>
    </div>

    <div class="text-end mt-3">
      <button type="submit" class="btn btn-success"><i class="fas fa-save me-1"></i> Save & Post Payment</button>
    </div>
  </form>
</div>

<script>
function debounceUpdateTotals() {
  clearTimeout(window._updateTimeout);
  window._updateTimeout = setTimeout(updateTotals, 200);
}

function updateTotals() {
  let paymentTotal = 0;
  let allocationTotal = 0;

  const paymentInput = document.querySelector("input[name='payment_amount']");
  const paymentValue = parseFloat(paymentInput.value) || 0;
  paymentTotal = paymentValue;
  paymentInput.classList.toggle("is-invalid", paymentValue < 0);

  document.querySelectorAll(".alloc").forEach(input => {
    const max = parseFloat(input.getAttribute("max"));
    const value = parseFloat(input.value) || 0;
    allocationTotal += value;
    input.classList.toggle("is-invalid", value > max || value < 0);
  });

  document.getElementById("allocatedTotal").textContent = `KSh ${allocationTotal.toLocaleString(undefined, { minimumFractionDigits: 2 })}`;
  document.getElementById("totalPaid").textContent = `KSh ${paymentTotal.toLocaleString(undefined, { minimumFractionDigits: 2 })}`;

  const overpay = paymentTotal - allocationTotal;
  if (overpay > 0) {
    document.getElementById("overpaymentSection").style.display = "block";
    document.getElementById("overpaymentAmount").textContent = `KSh ${overpay.toLocaleString(undefined, { minimumFractionDigits: 2 })}`;
  } else {
    document.getElementById("overpaymentSection").style.display = "none";
  }
}

function attachListeners() {
  document.querySelectorAll(".alloc, input[name='payment_amount']").forEach(input => {
    input.removeEventListener("input", debounceUpdateTotals);
    input.addEventListener("input", debounceUpdateTotals);
  });
}
attachListeners();
window.addEventListener("DOMContentLoaded", updateTotals);

document.getElementById("paymentForm").addEventListener("submit", function (e) {
  const allocations = [];
  document.querySelectorAll(".alloc").forEach(input => {
    const invoiceId = input.dataset.id;
    const amount = parseFloat(input.value) || 0;
    if (amount > 0) {
      allocations.push({ invoice_id: invoiceId, amount: amount, supplier_id: <?= $supplier_id ?> });
    }
  });

  if (allocations.length === 0) {
    alert("⚠️ Please allocate at least one invoice.");
    e.preventDefault();
    return;
  }

  const paymentAmount = parseFloat(document.querySelector("input[name='payment_amount']").value) || 0;
  if (paymentAmount <= 0) {
    alert("⚠️ Please enter a valid payment amount.");
    e.preventDefault();
    return;
  }

  if (paymentAmount < allocations.reduce((sum, alloc) => sum + alloc.amount, 0)) {
    alert("⚠️ Total allocated amount cannot exceed payment amount.");
    e.preventDefault();
    return;
  }

  document.getElementById("invoice_allocations_json").value = JSON.stringify(allocations);
});
</script>
<?php include '../../includes/footer.php'; ?>