<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

$user_role = $_SESSION['role'] ?? '';

// Filters
$branch_id = $_GET['branch_id'] ?? 0;
$from_date = $_GET['from_date'] ?? '';
$to_date = $_GET['to_date'] ?? '';

$conditions = [];
$params = [];
$types = '';

if ($branch_id > 0) {
    $conditions[] = "pct.branch_id = ?";
    $params[] = $branch_id;
    $types .= 'i';
}
if (!empty($from_date)) {
    $conditions[] = "pct.transaction_date >= ?";
    $params[] = $from_date;
    $types .= 's';
}
if (!empty($to_date)) {
    $conditions[] = "pct.transaction_date <= ?";
    $params[] = $to_date;
    $types .= 's';
}
$where = count($conditions) ? 'WHERE ' . implode(' AND ', $conditions) : '';

// Fetch data
$query = "
    SELECT pct.*, b.branch_name
    FROM petty_cash_transactions pct
    LEFT JOIN branches b ON pct.branch_id = b.id
    $where
    ORDER BY pct.transaction_date DESC, b.branch_name, pct.id DESC";
$stmt = $conn->prepare($query);
if ($params) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();

// Group by date then branch
$grouped = [];
$grand_total = 0;
while ($row = $result->fetch_assoc()) {
    $date = $row['transaction_date'];
    $branch = $row['branch_name'];
    $grouped[$date][$branch][] = $row;
    $grand_total += $row['amount'];
}

$branches = $conn->query("SELECT id, branch_name FROM branches ORDER BY branch_name ASC");
?>

<div class="main-content">
  <div class="d-flex justify-content-between align-items-center mb-4">
    <h4 class="fw-bold text-primary"><i class="fas fa-calendar-alt me-2"></i> Petty Cash by Date & Branch</h4>
    <a href="add.php" class="btn btn-primary"><i class="fas fa-plus-circle me-1"></i> Add Petty Cash</a>
  </div>

  <!-- Filters -->
  <div class="card shadow-sm mb-4">
    <div class="card-body">
      <form method="GET" class="row g-2">
        <div class="col-md-3">
          <select name="branch_id" class="form-select form-select-sm">
            <option value="">-- All Branches --</option>
            <?php while ($b = $branches->fetch_assoc()): ?>
              <option value="<?= $b['id'] ?>" <?= ($branch_id == $b['id']) ? 'selected' : '' ?>>
                <?= htmlspecialchars($b['branch_name']) ?>
              </option>
            <?php endwhile; ?>
          </select>
        </div>
        <div class="col-md-2">
          <input type="date" name="from_date" value="<?= htmlspecialchars($from_date) ?>" class="form-control form-control-sm">
        </div>
        <div class="col-md-2">
          <input type="date" name="to_date" value="<?= htmlspecialchars($to_date) ?>" class="form-control form-control-sm">
        </div>
        <div class="col-md-2">
          <button type="submit" class="btn btn-sm btn-primary w-100"><i class="fas fa-search me-1"></i> Filter</button>
        </div>
        <div class="col-md-2">
          <a href="list.php" class="btn btn-sm btn-secondary w-100"><i class="fas fa-sync-alt me-1"></i> Reset</a>
        </div>
      </form>
    </div>
  </div>

  <?php if (!empty($grouped)): ?>
    <div class="accordion" id="cashAccordion">
      <?php $acc = 0; foreach ($grouped as $date => $branches): ?>
        <?php
          $date_total = 0;
          foreach ($branches as $items) {
              $date_total += array_sum(array_column($items, 'amount'));
          }
          $is_overdue = strtotime($date) < strtotime('-3 days');
        ?>
        <div class="accordion-item mb-3 shadow-sm <?= $is_overdue ? 'border border-danger' : '' ?>">
          <h2 class="accordion-header" id="heading<?= $acc ?>">
            <button class="accordion-button <?= $acc > 0 ? 'collapsed' : '' ?>" type="button" data-bs-toggle="collapse" data-bs-target="#collapse<?= $acc ?>" aria-expanded="<?= $acc == 0 ? 'true' : 'false' ?>" aria-controls="collapse<?= $acc ?>">
              <strong><?= date('l, d M Y', strtotime($date)) ?></strong>
              <?php if ($is_overdue): ?>
                <span class="badge bg-danger ms-3">Overdue</span>
              <?php endif; ?>
              <span class="ms-2 text-success">Total: KSh <?= number_format($date_total, 2) ?></span>
            </button>
          </h2>
          <div id="collapse<?= $acc ?>" class="accordion-collapse collapse <?= $acc == 0 ? 'show' : '' ?>" data-bs-parent="#cashAccordion">
            <div class="accordion-body p-0">
              <?php foreach ($branches as $branch => $records): ?>
                <div class="px-3 pt-3">
                  <h6 class="text-primary border-bottom pb-1"><i class="fas fa-store me-1"></i><?= $branch ?> — KSh <?= number_format(array_sum(array_column($records, 'amount')), 2) ?></h6>
                </div>
                <div class="table-responsive px-3 mb-4">
                  <table class="table table-sm table-bordered align-middle mb-0">
                    <thead class="table-light text-center">
                      <tr>
                        <th>#</th>
                        <th>Ref No</th>
                        <th>Payee</th>
                        <th>Amount</th>
                        <th>Status</th>
                        <th>Collected By</th>
                        <th>Actions</th>
                      </tr>
                    </thead>
                    <tbody class="text-center">
                      <?php $i = 1; foreach ($records as $row): ?>
                        <tr>
                          <td><?= $i++ ?></td>
                          <td><?= htmlspecialchars($row['reference_no']) ?></td>
                          <td><?= htmlspecialchars($row['payee_name']) ?></td>
                          <td>KSh <?= number_format($row['amount'], 2) ?></td>
                          <td>
                            <span class="badge <?= $row['status'] == 'approved' ? 'bg-success' : 'bg-warning text-dark' ?>">
                              <?= ucfirst($row['status']) ?>
                            </span>
                          </td>
                          <td><?= htmlspecialchars($row['collected_by']) ?></td>
                          <td>
                            <div class="d-flex justify-content-center gap-1">
                              <?php if ($user_role === 'admin'): ?>
                                <a href="print_petty_cash.php?id=<?= $row['id'] ?>" class="btn btn-sm btn-outline-info" target="_blank"><i class="fas fa-print"></i></a>
                              <?php endif; ?>
                              <?php if ($row['status'] === 'pending'): ?>
                                <button type="button"
                                  class="btn btn-sm btn-success"
                                  data-bs-toggle="modal"
                                  data-bs-target="#approveModal"
                                  data-id="<?= $row['id'] ?>"
                                  data-ref="<?= htmlspecialchars($row['reference_no']) ?>"
                                  data-date="<?= htmlspecialchars($row['transaction_date']) ?>"
                                  data-branch="<?= htmlspecialchars($row['branch_name']) ?>"
                                  data-payee="<?= htmlspecialchars($row['payee_name']) ?>"
                                  data-amount="<?= number_format($row['amount'], 2) ?>"
                                  data-collected="<?= htmlspecialchars($row['collected_by']) ?>">
                                  <i class="fas fa-upload"></i>
                                </button>
                              <?php endif; ?>
                            </div>
                          </td>
                        </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                </div>
              <?php endforeach; ?>
            </div>
          </div>
        </div>
        <?php $acc++; ?>
      <?php endforeach; ?>
    </div>
    <div class="text-end mt-4 border-top pt-3">
      <h5 class="fw-bold text-success">Grand Total: KSh <?= number_format($grand_total, 2) ?></h5>
    </div>
  <?php else: ?>
    <div class="alert alert-info text-center">No transactions found for the selected filters.</div>
  <?php endif; ?>
</div>

<!-- Approval Modal -->
<div class="modal fade" id="approveModal" tabindex="-1" aria-labelledby="approveModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-lg">
    <form method="GET" action="upload_support.php" class="modal-content shadow border-0 rounded-3">
      <input type="hidden" name="id" id="approve_id">
      <div class="modal-header text-white" style="background-color: #1565c0;">
        <h5 class="modal-title" id="approveModalLabel"><i class="fas fa-check-circle me-2"></i> Approve Petty Cash</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <div id="approvalDetails" class="px-2">
          <div class="row g-2 mb-2">
            <div class="col-md-4"><strong>Ref No:</strong> <span id="modal_ref"></span></div>
            <div class="col-md-4"><strong>Date:</strong> <span id="modal_date"></span></div>
            <div class="col-md-4"><strong>Branch:</strong> <span id="modal_branch"></span></div>
          </div>
          <div class="row g-2 mb-2">
            <div class="col-md-4"><strong>Payee:</strong> <span id="modal_payee"></span></div>
            <div class="col-md-4"><strong>Amount:</strong> <span id="modal_amount"></span></div>
            <div class="col-md-4"><strong>Collected By:</strong> <span id="modal_collected"></span></div>
          </div>
          <div class="alert alert-warning mt-3 mb-0 small">
            ✅ Are you sure you want to approve this request? Once approved, it will be marked as complete.
          </div>
        </div>
      </div>
      <div class="modal-footer bg-light border-top">
        <button type="submit" class="btn btn-success"><i class="fas fa-check me-1"></i> Approve</button>
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
      </div>
    </form>
  </div>
</div>

<!-- Modal JS -->
<script>
document.addEventListener('DOMContentLoaded', function () {
  const modal = document.getElementById('approveModal');
  modal.addEventListener('show.bs.modal', function (event) {
    const btn = event.relatedTarget;
    modal.querySelector('#approve_id').value = btn.dataset.id;
    modal.querySelector('#modal_ref').textContent = btn.dataset.ref;
    modal.querySelector('#modal_date').textContent = btn.dataset.date;
    modal.querySelector('#modal_branch').textContent = btn.dataset.branch;
    modal.querySelector('#modal_payee').textContent = btn.dataset.payee;
    modal.querySelector('#modal_amount').textContent = 'KSh ' + btn.dataset.amount;
    modal.querySelector('#modal_collected').textContent = btn.dataset.collected;
  });
});
</script>
