<?php
/**
 * FILE: modules/reconcile/reconcile_cleared_cheques.php
 * Adds smart cheque-number extraction from Description (e.g. "CHQ No. 000407", "CHQ407", "CHQ000511 ...").
 * Matches with or without leading zeros.
 */

session_start();
require_once '../../config/db.php';

// ---- CSRF ----
function csrf_token(){ if(empty($_SESSION['csrf_token'])) $_SESSION['csrf_token']=bin2hex(random_bytes(16)); return $_SESSION['csrf_token']; }
function csrf_validate_or_die(){ $t=$_POST['csrf_token']??''; if(!$t || $t!==($_SESSION['csrf_token']??'')){ http_response_code(419); die('Invalid session (CSRF).'); }}

// ---- Helpers ----
function money_norm($v){
  $s=trim((string)$v); $neg=false;
  if(preg_match('/^\((.+)\)$/',$s,$m)){ $s=$m[1]; $neg=true; }
  $s=str_replace([' ',','],['',''],$s);
  if($s==='') return 0.0;
  $f=(float)$s;
  return $neg?- $f:$f;
}
function only_digits($s){ return preg_replace('/\D+/', '', (string)$s); }
function strip_leading_zeros($s){ return ltrim((string)$s,'0'); }
function html($s){ return htmlspecialchars((string)$s,ENT_QUOTES,'UTF-8'); }

/**
 * Try to extract cheque number from a free-text description.
 * Handles:
 *  - "CHQ No. 000407"
 *  - "CHQNo000407"
 *  - "CHQ407"
 *  - "INHouse CHQ000511 JUBILEE..."
 * Returns string digits (no spaces), possibly with leading zeros kept; caller can normalize.
 */
function extract_cheque_no_from_desc(?string $desc): string {
  $d = (string)$desc;

  // Common explicit forms like "CHQ No. 000407" / "Cheque No 000407"
  if (preg_match('/\b(?:CHQ|CHEQUE)\s*No\.?\s*0*(\d{3,})\b/i', $d, $m)) {
    return $m[1];
  }
  // Compact or spaced: "CHQ000511", "CHQ 000511", "CHQ407"
  if (preg_match('/\bCHQ\s*0*(\d{3,})\b/i', $d, $m)) {
    return $m[1];
  }
  // Fallback: any "...No 000407" near CHQ
  if (preg_match('/\b(?:No\.?|Number)\s*0*(\d{3,})\b.*\bCHQ\b/i', $d, $m)) {
    return $m[1];
  }
  // Last resort: take last 3–8 consecutive digits near "CHQ"
  if (preg_match('/CHQ[^0-9]{0,10}0?(\d{3,8})/i', $d, $m)) {
    return $m[1];
  }
  return '';
}

// ---- State ----
$step = $_POST['step'] ?? $_GET['step'] ?? 'upload';
$errors = [];
$info = [];

// ---- Upload CSV ----
if ($step==='upload' && $_SERVER['REQUEST_METHOD']==='POST'){
  csrf_validate_or_die();
  if (!isset($_FILES['csv']) || $_FILES['csv']['error']!==UPLOAD_ERR_OK){
    $errors[]='Please select a valid CSV file.'; $step='upload';
  }else{
    $fp=fopen($_FILES['csv']['tmp_name'],'r');
    if(!$fp){ $errors[]='Unable to read the uploaded CSV.'; $step='upload'; }
    else{
      $header=fgetcsv($fp);
      if(!$header){ $errors[]='CSV is empty or missing a header row.'; $step='upload'; }
      else{
        $rows=[]; while(($r=fgetcsv($fp))!==false){ $rows[]=$r; } fclose($fp);
        $_SESSION['recon_csv']=['name'=>($_FILES['csv']['name']??'statement.csv'),'header'=>$header,'rows'=>$rows];
        $step='map';
      }
    }
  }
}

// ---- Map columns guard ----
if ($step==='map' && empty($_SESSION['recon_csv'])){ $errors[]='No CSV in session. Please upload again.'; $step='upload'; }

// ---- Build preview buckets ----
$preview=['perfect'=>[],'status_mismatch'=>[],'amount_mismatch'=>[],'not_found'=>[]];
$mapping=['date'=>null,'desc'=>null,'cheque_no'=>null,'amount'=>null,'drcr'=>null];

// ---- Preview ----
if ($step==='preview' && $_SERVER['REQUEST_METHOD']==='POST'){
  csrf_validate_or_die();
  if (empty($_SESSION['recon_csv'])){ $errors[]='Session expired. Re-upload CSV.'; $step='upload'; }
  else{
    $csv=$_SESSION['recon_csv']; $header=$csv['header']; $rows=$csv['rows'];

    $mapping['date']      = isset($_POST['map_date'])      ? (int)$_POST['map_date']      : -1;
    $mapping['desc']      = isset($_POST['map_desc'])      ? (int)$_POST['map_desc']      : -1;
    $mapping['cheque_no'] = isset($_POST['map_cheque_no']) ? (int)$_POST['map_cheque_no'] : -1;
    $mapping['amount']    = isset($_POST['map_amount'])    ? (int)$_POST['map_amount']    : -1;
    $mapping['drcr']      = isset($_POST['map_drcr'])      ? (int)$_POST['map_drcr']      : -1;

    if ($mapping['amount']<0){ $errors[]='Please map Amount column.'; $step='map'; }
    elseif ($mapping['cheque_no']<0 && $mapping['desc']<0){ $errors[]='Map Cheque Number OR Description (for auto-extract).'; $step='map'; }
    else{
      // Prepared statements:
      // 1) exact space-stripped equality on reference fields
      $stmtExact = $conn->prepare("
        SELECT id, supplier_id, amount, payment_date, payment_reference, reference_no, method, cheque_status, remarks
        FROM payments
        WHERE method='cheque'
          AND (REPLACE(payment_reference,' ','') = ? OR REPLACE(reference_no,' ','') = ?)
        LIMIT 50
      ");
      // 2) LIKE match to pull candidates; we’ll validate digits in PHP to avoid false hits
      $stmtLike = $conn->prepare("
        SELECT id, supplier_id, amount, payment_date, payment_reference, reference_no, method, cheque_status, remarks
        FROM payments
        WHERE method='cheque'
          AND (payment_reference LIKE ? OR reference_no LIKE ?)
        ORDER BY id DESC
        LIMIT 200
      ");

      foreach($rows as $line_no=>$r){
        $bank_date = ($mapping['date']>=0 && isset($r[$mapping['date']])) ? trim($r[$mapping['date']]) : '';
        $bank_desc = ($mapping['desc']>=0 && isset($r[$mapping['desc']])) ? trim($r[$mapping['desc']]) : '';
        $raw_cheque= ($mapping['cheque_no']>=0 && isset($r[$mapping['cheque_no']])) ? trim($r[$mapping['cheque_no']]) : '';
        $raw_amount= ($mapping['amount']>=0 && isset($r[$mapping['amount']])) ? trim($r[$mapping['amount']]) : '';

        $bank_amt = abs(money_norm($raw_amount));
        // determine cheque from explicit column or description extraction
        $bank_chq = $raw_cheque !== '' ? $raw_cheque : extract_cheque_no_from_desc($bank_desc);
        $bank_chq_digits = only_digits($bank_chq);

        if ($bank_chq_digits===''){
          // not a cheque row
          continue;
        }

        // Search strategy:
        // A) exact match with/without leading zeros (spaces removed)
        $needleA = $bank_chq_digits;                 // e.g. 000407 or 407
        $needleB = strip_leading_zeros($needleA);    // e.g. 407
        if ($needleB==='') $needleB='0';

        $found = [];
        $try = [$needleA, $needleB];
        $seen_ids = [];

        foreach ($try as $nd){
          $stmtExact->bind_param('ss',$nd,$nd);
          $stmtExact->execute();
          $res=$stmtExact->get_result();
          while($p=$res->fetch_assoc()){
            if(isset($seen_ids[$p['id']])) continue;
            $seen_ids[$p['id']]=1;
            $found[]=$p;
          }
          if(!empty($found)) break; // good enough
        }

        // B) if still nothing, fallback to LIKE and validate by digits
        if (empty($found)){
          $like = '%'.$needleB.'%';
          $stmtLike->bind_param('ss',$like,$like);
          $stmtLike->execute();
          $res=$stmtLike->get_result();
          while($p=$res->fetch_assoc()){
            $r1 = only_digits($p['payment_reference'] ?? '');
            $r2 = only_digits($p['reference_no'] ?? '');
            if ($r1===$needleA || $r1===$needleB || $r2===$needleA || $r2===$needleB){
              if(!isset($seen_ids[$p['id']])){
                $seen_ids[$p['id']]=1;
                $found[]=$p;
              }
            }
          }
        }

        if (empty($found)){
          $preview['not_found'][] = [
            'line'=>$line_no+2,
            'bank_date'=>$bank_date,
            'bank_desc'=>$bank_desc,
            'bank_cheque_no'=>$bank_chq ?: $bank_chq_digits,
            'bank_amount'=>$bank_amt,
            'hint'=>'Cheque not found in payments (checked ref & alt forms).'
          ];
          continue;
        }

        // Choose best by amount proximity
        usort($found, fn($a,$b)=>abs(((float)$a['amount'])-$bank_amt)<=>abs(((float)$b['amount'])-$bank_amt));
        $best=$found[0];

        $amount_equal = (abs(((float)$best['amount']) - $bank_amt) < 0.005);
        $status = strtolower((string)$best['cheque_status']);

        $payload = [
          'line'=>$line_no+2,
          'bank_date'=>$bank_date,
          'bank_desc'=>$bank_desc,
          'bank_cheque_no'=>($raw_cheque!==''?$raw_cheque:($bank_chq!==''?$bank_chq:$bank_chq_digits)),
          'bank_amount'=>$bank_amt,
          'payment_id'=>(int)$best['id'],
          'payment_amount'=>(float)$best['amount'],
          'payment_reference'=>$best['payment_reference'],
          'reference_no'=>$best['reference_no'],
          'payment_date'=>$best['payment_date'],
          'cheque_status'=>$best['cheque_status'],
        ];

        if ($amount_equal){
          if (in_array($status,['cleared','cleared '])){
            $preview['perfect'][]=$payload;
          } else {
            $preview['status_mismatch'][]=$payload;
          }
        } else {
          $preview['amount_mismatch'][]=$payload;
        }
      }

      $stmtExact->close(); $stmtLike->close();
      $_SESSION['recon_preview']=$preview;
    }
  }
}

// ---- Commit (logs) ----
$committed=0;
if ($step==='commit' && $_SERVER['REQUEST_METHOD']==='POST'){
  csrf_validate_or_die();
  $uid=(int)($_SESSION['user_id']??0);
  if($uid<=0){ http_response_code(403); die('Please log in first.'); }
  if(empty($_SESSION['recon_preview'])){ $errors[]='Nothing to commit. Run a preview first.'; $step='upload'; }
  else{
    $to_commit=$_POST['commit_ids']??[];
    if(!is_array($to_commit) || empty($to_commit)){ $errors[]='No rows selected.'; $step='preview'; }
    else{
      $conn->query("
        CREATE TABLE IF NOT EXISTS cheque_reconciliations (
          id INT AUTO_INCREMENT PRIMARY KEY,
          payment_id INT NOT NULL,
          bank_date DATE NULL,
          bank_description VARCHAR(255) NULL,
          bank_cheque_no VARCHAR(100) NOT NULL,
          bank_amount DECIMAL(12,2) NOT NULL,
          created_by INT NOT NULL,
          created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
          UNIQUE KEY uq_payment_once (payment_id),
          INDEX idx_bank_fields (bank_cheque_no, bank_amount)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
      ");

      $perfectLut=[]; foreach($_SESSION['recon_preview']['perfect'] as $r){ $perfectLut[$r['payment_id'].'|'.$r['line']]=$r; }

      $stmt=$conn->prepare("INSERT IGNORE INTO cheque_reconciliations (payment_id,bank_date,bank_description,bank_cheque_no,bank_amount,created_by) VALUES (?,?,?,?,?,?)");
      foreach($to_commit as $key){
        if(!isset($perfectLut[$key])) continue;
        $r=$perfectLut[$key];
        $d=null; if(!empty($r['bank_date'])){ $ts=strtotime($r['bank_date']); if($ts) $d=date('Y-m-d',$ts); }
        $stmt->bind_param('isssdi',$r['payment_id'],$d,$r['bank_desc'],$r['bank_cheque_no'],$r['bank_amount'],$uid);
        if($stmt->execute()) $committed++;
      }
      $stmt->close();
      $info[]="Committed $committed reconciliation row(s).";
      $step='done';
    }
  }
}
?>
<!doctype html>
<html lang="en" data-bs-theme="light">
<head>
  <meta charset="utf-8">
  <title>Cheque Reconciliation (Cleared Cheques)</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <meta name="csrf-token" content="<?= html(csrf_token()) ?>">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .sticky-head{position:sticky;top:0;z-index:1}
    .badge-dot{display:inline-block;width:10px;height:10px;border-radius:50%;margin-right:6px}
    .bg-perfect{background:#e8f5e9}
    .bg-status-mismatch{background:#fff8e1}
    .bg-amount-mismatch{background:#ffebee}
    .bg-not-found{background:#eceff1}
    .small{font-size:.875rem}
    .pre{white-space:pre-wrap}
  </style>
</head>
<body class="p-3">
  <div class="container-fluid">
    <h3 class="fw-bold mb-3">🧾 Cheque Reconciliation — Cleared Cheques</h3>

    <?php if($errors): ?><div class="alert alert-danger"><ul class="m-0"><?php foreach($errors as $e): ?><li><?=html($e)?></li><?php endforeach; ?></ul></div><?php endif; ?>
    <?php if($info): ?><div class="alert alert-success"><?php foreach($info as $i): ?><?=html($i)?><br><?php endforeach; ?></div><?php endif; ?>

    <?php if($step==='upload'): ?>
      <div class="card shadow-sm">
        <div class="card-body">
          <form method="post" enctype="multipart/form-data">
            <input type="hidden" name="csrf_token" value="<?=html(csrf_token())?>">
            <input type="hidden" name="step" value="upload">
            <div class="mb-3">
              <label class="form-label fw-semibold">Bank Statement (CSV)</label>
              <input type="file" name="csv" accept=".csv,text/csv" class="form-control" required>
              <div class="form-text">Map Cheque No OR just Description — we’ll auto-extract numbers like “CHQ No. 000407”, “CHQ407”, “CHQ000511”.</div>
            </div>
            <button class="btn btn-primary">Upload & Continue</button>
          </form>
        </div>
      </div>

    <?php elseif($step==='map'):
      $csv=$_SESSION['recon_csv']; $header=$csv['header'];
      $opts=function($name) use($header){
        $html='<option value="-1">-- Not Applicable --</option>';
        foreach($header as $i=>$h){
          $label=trim($h)===''?'Column #'.($i+1):$h; $sel='';
          $lh=strtolower($h);
          if($name==='map_cheque_no' && preg_match('/(cheque|chq|ref|reference|number|no)/i',$lh)) $sel=' selected';
          if($name==='map_amount' && preg_match('/(amount|amt|value|ksh|kes)/i',$lh)) $sel=' selected';
          if($name==='map_date' && preg_match('/(date|value date|txn date|posting date)/i',$lh)) $sel=' selected';
          if($name==='map_desc' && preg_match('/(description|narration|details|memo|particular)/i',$lh)) $sel=' selected';
          if($name==='map_drcr' && preg_match('/(dr\/cr|type|drcr|dc)/i',$lh)) $sel=' selected';
          $html.='<option value="'.$i.'"'.$sel.'>'.htmlspecialchars($label).'</option>';
        }
        return $html;
      };
    ?>
      <div class="card shadow-sm">
        <div class="card-body">
          <h5 class="fw-bold mb-3">Map CSV Columns</h5>
          <form method="post" class="row g-3">
            <input type="hidden" name="csrf_token" value="<?=html(csrf_token())?>">
            <input type="hidden" name="step" value="preview">

            <div class="col-md-4"><label class="form-label">Date Column</label><select name="map_date" class="form-select"><?= $opts('map_date') ?></select></div>
            <div class="col-md-4"><label class="form-label">Description Column (auto-extract CHQ…)</label><select name="map_desc" class="form-select"><?= $opts('map_desc') ?></select></div>
            <div class="col-md-4"><label class="form-label">Cheque Number Column (optional)</label><select name="map_cheque_no" class="form-select"><?= $opts('map_cheque_no') ?></select></div>
            <div class="col-md-4"><label class="form-label">Amount Column <span class="text-danger">*</span></label><select name="map_amount" class="form-select" required><?= $opts('map_amount') ?></select></div>
            <div class="col-md-4"><label class="form-label">DR/CR Column (optional)</label><select name="map_drcr" class="form-select"><?= $opts('map_drcr') ?></select></div>

            <div class="col-12">
              <button class="btn btn-primary">Preview Matches</button>
              <a href="?step=upload" class="btn btn-outline-secondary">Re-upload CSV</a>
            </div>
          </form>
        </div>
      </div>

    <?php elseif($step==='preview' && isset($_SESSION['recon_preview'])):
      $preview=$_SESSION['recon_preview'];
    ?>
      <div class="mb-3"><a href="?step=upload" class="btn btn-sm btn-outline-secondary">⬅ Upload another CSV</a></div>

      <div class="row g-3">
        <div class="col-12 col-xl-6">
          <div class="card shadow-sm">
            <div class="card-header bg-success-subtle"><strong>✅ Perfect Matches (Already Cleared)</strong></div>
            <div class="card-body p-0">
              <?php if(empty($preview['perfect'])): ?>
                <div class="p-3 text-muted">No perfect matches found.</div>
              <?php else: ?>
                <form method="post" class="p-0 m-0">
                  <input type="hidden" name="csrf_token" value="<?=html(csrf_token())?>">
                  <input type="hidden" name="step" value="commit">
                  <div class="table-responsive">
                    <table class="table table-sm align-middle mb-0">
                      <thead class="table-light sticky-head">
                        <tr>
                          <th style="width:36px"><input type="checkbox" onclick="document.querySelectorAll('.ck-perfect').forEach(x=>x.checked=this.checked)"></th>
                          <th>Bank Date</th><th>Cheque No</th>
                          <th class="text-end">Bank Amt</th><th class="text-end">Pay Amt</th>
                          <th>Ref</th><th>Status</th>
                        </tr>
                      </thead>
                      <tbody class="bg-perfect">
                        <?php foreach($preview['perfect'] as $r): $key=$r['payment_id'].'|'.$r['line']; ?>
                          <tr>
                            <td><input class="form-check-input ck-perfect" type="checkbox" name="commit_ids[]" value="<?=html($key)?>" checked></td>
                            <td class="small"><?=html($r['bank_date'])?></td>
                            <td class="small"><?=html($r['bank_cheque_no'])?></td>
                            <td class="text-end small"><?=number_format($r['bank_amount'],2)?></td>
                            <td class="text-end small"><?=number_format($r['payment_amount'],2)?></td>
                            <td class="small"><?=html($r['reference_no'] ?: $r['payment_reference'])?><br><span class="text-muted">PID #<?= (int)$r['payment_id']?></span></td>
                            <td><span class="badge text-bg-success">Cleared</span></td>
                          </tr>
                        <?php endforeach; ?>
                      </tbody>
                    </table>
                  </div>
                  <div class="p-2 border-top"><button class="btn btn-success btn-sm">Log Reconciliations</button></div>
                </form>
              <?php endif; ?>
            </div>
          </div>
        </div>

        <div class="col-12 col-xl-6">
          <div class="card shadow-sm mb-3">
            <div class="card-header bg-warning-subtle"><strong>🟨 Status Mismatch (Amount ok, Status not Cleared)</strong></div>
            <div class="card-body p-0">
              <?php if(empty($preview['status_mismatch'])): ?>
                <div class="p-3 text-muted">None.</div>
              <?php else: ?>
                <div class="table-responsive">
                  <table class="table table-sm align-middle mb-0">
                    <thead class="table-light sticky-head">
                      <tr><th>Bank Date</th><th>Cheque No</th><th class="text-end">Bank Amt</th><th class="text-end">Pay Amt</th><th>Ref</th><th>Status</th></tr>
                    </thead>
                    <tbody class="bg-status-mismatch">
                      <?php foreach($preview['status_mismatch'] as $r): ?>
                        <tr>
                          <td class="small"><?=html($r['bank_date'])?></td>
                          <td class="small"><?=html($r['bank_cheque_no'])?></td>
                          <td class="text-end small"><?=number_format($r['bank_amount'],2)?></td>
                          <td class="text-end small"><?=number_format($r['payment_amount'],2)?></td>
                          <td class="small"><?=html($r['reference_no'] ?: $r['payment_reference'])?></td>
                          <td><span class="badge text-bg-warning"><?=html($r['cheque_status'] ?: 'N/A')?></span></td>
                        </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                </div>
                <div class="p-2 border-top small text-muted">If these are truly cleared in bank, update <code>cheque_status</code> internally.</div>
              <?php endif; ?>
            </div>
          </div>

          <div class="card shadow-sm mb-3">
            <div class="card-header bg-danger-subtle"><strong>🟥 Amount Mismatch (Cheque found, different amount)</strong></div>
            <div class="card-body p-0">
              <?php if(empty($preview['amount_mismatch'])): ?>
                <div class="p-3 text-muted">None.</div>
              <?php else: ?>
                <div class="table-responsive">
                  <table class="table table-sm align-middle mb-0">
                    <thead class="table-light sticky-head">
                      <tr><th>Bank Date</th><th>Cheque No</th><th class="text-end">Bank Amt</th><th class="text-end">Pay Amt</th><th>Ref</th><th>Status</th></tr>
                    </thead>
                    <tbody class="bg-amount-mismatch">
                      <?php foreach($preview['amount_mismatch'] as $r): ?>
                        <tr>
                          <td class="small"><?=html($r['bank_date'])?></td>
                          <td class="small"><?=html($r['bank_cheque_no'])?></td>
                          <td class="text-end small"><?=number_format($r['bank_amount'],2)?></td>
                          <td class="text-end small"><?=number_format($r['payment_amount'],2)?></td>
                          <td class="small"><?=html($r['reference_no'] ?: $r['payment_reference'])?></td>
                          <td><span class="badge text-bg-secondary"><?=html($r['cheque_status'] ?: 'N/A')?></span></td>
                        </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                </div>
              <?php endif; ?>
            </div>
          </div>

          <div class="card shadow-sm">
            <div class="card-header bg-secondary-subtle"><strong>⬜ Not Found (Cheque not in payments)</strong></div>
            <div class="card-body p-0">
              <?php if(empty($preview['not_found'])): ?>
                <div class="p-3 text-muted">None.</div>
              <?php else: ?>
                <div class="table-responsive">
                  <table class="table table-sm align-middle mb-0">
                    <thead class="table-light sticky-head">
                      <tr><th>Bank Date</th><th>Cheque No</th><th>Description</th><th class="text-end">Bank Amt</th><th>Line</th></tr>
                    </thead>
                    <tbody class="bg-not-found">
                      <?php foreach($preview['not_found'] as $r): ?>
                        <tr>
                          <td class="small"><?=html($r['bank_date'])?></td>
                          <td class="small"><?=html($r['bank_cheque_no'])?></td>
                          <td class="small pre"><?=html($r['bank_desc'])?></td>
                          <td class="text-end small"><?=number_format($r['bank_amount'],2)?></td>
                          <td class="small text-muted">#<?= (int)$r['line']?></td>
                        </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                </div>
                <div class="p-2 border-top small text-muted">Check if these were recorded under different references or another bank account.</div>
              <?php endif; ?>
            </div>
          </div>
        </div>
      </div>

    <?php elseif($step==='done'): ?>
      <div class="alert alert-success">Reconciliation log complete. You can upload another statement or review your logs.</div>
      <div class="d-flex gap-2">
        <a href="?step=upload" class="btn btn-primary">Upload Another CSV</a>
        <a href="javascript:history.back()" class="btn btn-outline-secondary">Back</a>
      </div>
    <?php endif; ?>
  </div>
</body>
</html>
