<?php
session_start();
require_once '../../config/db.php';

// Log script version
error_log("Using edit.php version: c7f8d2a9-3b5e-4c1a-9e8b-6d4e2f1c0b7d");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = '../logs';
$log_file = "$log_dir/edit_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Check if national_id is provided and valid
if (!isset($_GET['national_id']) || !preg_match('/^\d{7,9}$/', $_GET['national_id'])) {
    $_SESSION['error'] = 'Invalid or missing National ID.';
    error_log("[$timestamp] Invalid or missing national_id: " . ($_GET['national_id'] ?? 'not provided'));
    header('Location: list.php');
    exit;
}

$national_id = $_GET['national_id'];

// Fetch staff record
$query = "
    SELECT national_id, first_name, last_name, email, phone, branch_id, role, status
    FROM staff WHERE national_id = ?
";
$stmt = $conn->prepare($query);
$stmt->bind_param('s', $national_id);
$stmt->execute();
$staff = $stmt->get_result()->fetch_assoc();

if (!$staff) {
    $_SESSION['error'] = 'Staff not found for National ID: ' . htmlspecialchars($national_id);
    error_log("[$timestamp] Staff not found for national_id: $national_id");
    header('Location: list.php');
    exit;
}

// Fetch branches
$branches = [];
try {
    $branches_query = "SELECT id, branch_name FROM branches ORDER BY branch_name";
    $branches_result = $conn->query($branches_query);
    while ($row = $branches_result->fetch_assoc()) {
        $branches[] = $row;
    }
    error_log("[$timestamp] Fetched " . count($branches) . " branches");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($branches) . " branches\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch branches failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch branches failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load branches: ' . $e->getMessage();
}

// Define roles (consistent with database)
$roles = [
    'manager', 'supervisor', 'accountant', 'chief_cashier', 'cashier', 'line_manager',
    'store_clerk', 'security', 'baker', 'receiving_clerk', 'general', 'packer'
];

// Display session messages
$success = $_SESSION['success'] ?? '';
$error = $_SESSION['error'] ?? '';
unset($_SESSION['success'], $_SESSION['error']);

// Close connection
$conn->close();

// Include layout
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Staff - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body { background-color: #f8f9fa; font-size: 14px; }
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); border: 1px solid #e0e0e0; }
        .card-header { font-size: 18px; background: linear-gradient(to right, #007bff, #00aaff); }
        .form-control, .form-select { 
            font-size: 14px; 
            border: 1px solid #007bff; 
            border-radius: 5px; 
            transition: border-color 0.3s ease; 
        }
        .form-control:hover, .form-select:hover { border-color: #0056b3; }
        .form-control:disabled { background-color: #e9ecef; }
        .form-label { font-weight: 500; color: #333; margin-bottom: 8px; }
        .btn-primary { 
            background: linear-gradient(to right, #007bff, #00aaff); 
            border: none; 
            border-radius: 5px; 
            padding: 10px 20px; 
            box-shadow: 0 2px 8px rgba(0,0,0,0.2); 
            transition: all 0.3s ease; 
        }
        .btn-primary:hover { 
            background: linear-gradient(to right, #0056b3, #0088cc); 
            box-shadow: 0 4px 12px rgba(0,0,0,0.3); 
        }
        .btn-secondary { 
            background-color: #6c757d; 
            border-color: #6c757d; 
            border-radius: 5px; 
        }
        .btn-secondary:hover { 
            background-color: #5a6268; 
            border-color: #545b62; 
        }
        .alert { border-radius: 8px; font-size: 14px; }
        .footer { text-align: center; margin-top: 30px; font-size: 13px; color: #888; }
        .invalid-feedback { font-size: 12px; }
        @media (max-width: 576px) {
            .card-body { padding: 15px; }
            .btn { width: 100%; margin-bottom: 10px; }
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <?php if ($success): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="fas fa-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if ($error): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="fas fa-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>

        <div class="card mb-4">
            <div class="card-header text-white text-center"><strong>Edit Staff (National ID: <?php echo htmlspecialchars($national_id); ?>)</strong></div>
            <div class="card-body">
                <form action="save_staff.php" method="POST" id="staffForm" novalidate>
                    <input type="hidden" name="national_id" value="<?php echo htmlspecialchars($national_id); ?>">
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label class="form-label"><strong>First Name</strong></label>
                            <input type="text" class="form-control" name="first_name" value="<?php echo htmlspecialchars($staff['first_name']); ?>" required pattern="[A-Za-z\s]{1,50}" title="First name must be 1-50 letters or spaces">
                            <div class="invalid-feedback">First name must be 1-50 letters or spaces.</div>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label"><strong>Last Name</strong></label>
                            <input type="text" class="form-control" name="last_name" value="<?php echo htmlspecialchars($staff['last_name']); ?>" required pattern="[A-Za-z\s]{1,50}" title="Last name must be 1-50 letters or spaces">
                            <div class="invalid-feedback">Last name must be 1-50 letters or spaces.</div>
                        </div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label class="form-label"><strong>National ID</strong></label>
                            <input type="text" class="form-control" value="<?php echo htmlspecialchars($staff['national_id']); ?>" disabled>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label"><strong>Email (Optional)</strong></label>
                            <input type="email" class="form-control" name="email" value="<?php echo htmlspecialchars($staff['email'] ?? ''); ?>" pattern="[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}" title="Enter a valid email address">
                            <div class="invalid-feedback">Enter a valid email address.</div>
                        </div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label class="form-label"><strong>Phone (Optional)</strong></label>
                            <input type="text" class="form-control" name="phone" value="<?php echo htmlspecialchars($staff['phone'] ?? ''); ?>" pattern="\+?[0-9]{10,15}" title="Phone number must be 10-15 digits, optionally starting with +">
                            <div class="invalid-feedback">Phone number must be 10-15 digits, optionally starting with +.</div>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label"><strong>Branch</strong></label>
                            <select class="form-select" name="branch_id" required>
                                <option value="">Select Branch</option>
                                <?php foreach ($branches as $branch): ?>
                                    <option value="<?php echo $branch['id']; ?>" <?php echo $staff['branch_id'] == $branch['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($branch['branch_name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <div class="invalid-feedback">Please select a branch.</div>
                        </div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label class="form-label"><strong>Role</strong></label>
                            <select class="form-select" name="role" required>
                                <option value="">Select Role</option>
                                <?php foreach ($roles as $role): ?>
                                    <option value="<?php echo $role; ?>" <?php echo $staff['role'] == $role ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars(ucfirst($role)); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <div class="invalid-feedback">Please select a role.</div>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label"><strong>Status</strong></label>
                            <select class="form-select" name="status" required>
                                <option value="active" <?php echo $staff['status'] == 'active' ? 'selected' : ''; ?>>Active</option>
                                <option value="inactive" <?php echo $staff['status'] == 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                            </select>
                            <div class="invalid-feedback">Please select a status.</div>
                        </div>
                    </div>
                    <div class="text-end mt-3">
                        <button type="submit" class="btn btn-primary"><i class="fas fa-save me-2"></i>Update Staff</button>
                        <a href="list.php" class="btn btn-secondary"><i class="fas fa-arrow-left me-2"></i>Cancel</a>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <div class="footer">
        © <?php echo date("Y"); ?> SmartPay360. All rights reserved.
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Client-side form validation
        (function () {
            'use strict';
            const form = document.getElementById('staffForm');
            form.addEventListener('submit', function (event) {
                if (!form.checkValidity()) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add('was-validated');
                const submitButton = form.querySelector('button[type="submit"]');
                if (form.dataset.submitted) {
                    event.preventDefault();
                } else {
                    form.dataset.submitted = true;
                    submitButton.disabled = true;
                    submitButton.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Updating...';
                }
            }, false);
        })();
    </script>
</body>
</html>