<?php
require_once '../../config/db.php';

$csvFile = __DIR__ . '/DELTA SUPERSTORES STAFF - Form responses 1.csv';
$photoDir = __DIR__ . '/../../uploads/staff_photos';

if (!file_exists($csvFile)) die("❌ File not found.");
if (!is_dir($photoDir)) mkdir($photoDir, 0755, true);

$row = 0;
$imported = 0;
$skipped = 0;
$handle = fopen($csvFile, 'r');
$header = fgetcsv($handle); // skip header

$conn->begin_transaction();

try {
    while (($data = fgetcsv($handle)) !== false) {
        $row++;
        [$timestamp, $full_name, $national_id, $phone, $gender, $branch_name, $department, $years, $photo_url] = $data;

        $names = explode(' ', trim($full_name), 2);
        $first_name = $names[0] ?? '';
        $last_name = $names[1] ?? '';

        // Find branch_id
        $stmt = $conn->prepare("SELECT id FROM branches WHERE branch_name = ?");
        $stmt->bind_param("s", $branch_name);
        $stmt->execute();
        $stmt->bind_result($branch_id);
        $stmt->fetch();
        $stmt->close();

        if (!$branch_id) {
            error_log("Skipping row $row: Unknown branch '$branch_name'");
            $skipped++;
            continue;
        }

        // Check for duplicate national_id
        $stmt = $conn->prepare("SELECT COUNT(*) FROM staff WHERE national_id = ?");
        $stmt->bind_param("s", $national_id);
        $stmt->execute();
        $stmt->bind_result($exists);
        $stmt->fetch();
        $stmt->close();

        if ($exists > 0) {
            $skipped++;
            continue;
        }

        // Generate IDs and defaults
        $staff_id = 'STAFF-' . date('Ymd') . '-' . rand(1000, 9999);
        $role = strtolower(str_replace(' ', '_', trim($department)));
        $status = 'active';
        $created_by = 1;
        $photo_filename = null;

        // Download photo
        if ($photo_url && preg_match('/[\w-]{25,}/', $photo_url, $match)) {
            $file_id = $match[0];
            $photo_filename = "staff-$national_id.jpg";
            $save_path = "$photoDir/$photo_filename";

            // Download from Drive (export endpoint)
            $img = @file_get_contents("https://drive.google.com/uc?export=download&id=$file_id");
            if ($img !== false) {
                file_put_contents($save_path, $img);
            } else {
                $photo_filename = null;
            }
        }

        // Insert staff
        $stmt = $conn->prepare("
            INSERT INTO staff (
                staff_id, first_name, last_name, national_id, phone,
                branch_id, role, status, created_by, created_at, photo
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?)
        ");
        $stmt->bind_param(
            "sssssssiis",
            $staff_id, $first_name, $last_name, $national_id, $phone,
            $branch_id, $role, $status, $created_by, $photo_filename
        );
        $stmt->execute();
        $imported++;
    }

    fclose($handle);
    $conn->commit();
    echo "✅ Imported $imported staff. Skipped $skipped.";
} catch (Exception $e) {
    $conn->rollback();
    echo "❌ Import failed: " . $e->getMessage();
}
