```php
<?php
session_start();
require_once '../../config/db.php';

// Log script version
error_log("Using off_scheduling.php version: d6y7z8a9-5b6c-1d2e-7f8g-9h0i1j2k3l4");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = '../logs';
$log_file = "$log_dir/off_scheduling_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $staff_ids = $_POST['staff_ids'] ?? [];
    $start_date = $_POST['start_date'] ?? '';
    $end_date = $_POST['end_date'] ?? '';
    $reason = trim($_POST['reason'] ?? '');
    $created_by = 1; // Hardcoded user_id (TODO: Replace with session user)

    if (empty($staff_ids) || !$start_date || !$end_date || !$reason) {
        $_SESSION['error'] = 'All fields are required.';
        header('Location: off_scheduling.php');
        exit;
    }

    $conn->begin_transaction();
    try {
        foreach ($staff_ids as $staff_id) {
            $query = "SELECT COUNT(*) as count FROM staff WHERE staff_id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param('s', $staff_id);
            $stmt->execute();
            if ($stmt->get_result()->fetch_assoc()['count'] == 0) {
                throw new Exception("Invalid staff ID: $staff_id");
            }

            $current_date = new DateTime($start_date);
            $end = new DateTime($end_date);
            while ($current_date <= $end) {
                $date_str = $current_date->format('Y-m-d');
                $query = "
                    INSERT INTO schedules (staff_id, branch_id, schedule_date, shift_type, reason, created_by, created_at)
                    SELECT staff_id, branch_id, ?, 'off', ?, ?, NOW()
                    FROM staff WHERE staff_id = ?
                ";
                $stmt = $conn->prepare($query);
                $stmt->bind_param('ssis', $date_str, $reason, $created_by, $staff_id);
                $stmt->execute();
                $current_date->modify('+1 day');
            }
        }

        $conn->commit();
        error_log("[$timestamp] Off-schedule added for " . count($staff_ids) . " staff");
        file_put_contents($log_file, "[$timestamp] Off-schedule added for " . count($staff_ids) . " staff\n", FILE_APPEND);
        $_SESSION['success'] = 'Off-schedule added for ' . count($staff_ids) . ' staff.';
        header('Location: off_scheduling.php');
        exit;
    } catch (Exception $e) {
        $conn->rollback();
        error_log("[$timestamp] Off-scheduling failed: " . $e->getMessage());
        file_put_contents($log_file, "[$timestamp] Off-scheduling failed: " . $e->getMessage() . "\n", FILE_APPEND);
        $_SESSION['error'] = 'Failed to add off-schedule: ' . $e->getMessage();
        header('Location: off_scheduling.php');
        exit;
    }
}

// Fetch staff
$staff = [];
try {
    $query = "SELECT staff_id, first_name, last_name FROM staff ORDER BY first_name, last_name";
    $result = $conn->query($query);
    while ($row = $result->fetch_assoc()) {
        $staff[] = $row;
    }
    error_log("[$timestamp] Fetched " . count($staff) . " staff");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($staff) . " staff\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch staff failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch staff failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load staff.';
}

// Close connection
$conn->close();

// Include layout
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Off-Scheduling - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet">
    <style>
        .card { border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .form-label { font-weight: 500; }
        .btn-primary { background-color: #007bff; border-color: #007bff; }
        .btn-primary:hover { background-color: #0056b3; border-color: #0056b3; }
        .select2-container { width: 100% !important; }
        .alert { border-radius: 8px; }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="mb-4">Off-Scheduling</h2>
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <div class="card p-4">
            <div class="card-body">
                <h5 class="card-title">Schedule Off-Days</h5>
                <form id="offScheduleForm" method="POST">
                    <div class="mb-3">
                        <label for="staff_ids" class="form-label">Select Staff <i class="fas fa-info-circle" data-bs-toggle="tooltip" title="Select multiple staff for off-scheduling"></i></label>
                        <select name="staff_ids[]" id="staff_ids" class="form-select" multiple required>
                            <?php foreach ($staff as $s): ?>
                                <option value="<?php echo $s['staff_id']; ?>">
                                    <?php echo htmlspecialchars($s['first_name'] . ' ' . $s['last_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="start_date" class="form-label">Start Date</label>
                            <input type="date" name="start_date" id="start_date" class="form-control" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="end_date" class="form-label">End Date</label>
                            <input type="date" name="end_date" id="end_date" class="form-control" required>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="reason" class="form-label">Reason</label>
                        <input type="text" name="reason" id="reason" class="form-control" placeholder="e.g., Vacation, Sick Leave" required>
                    </div>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#confirmModal"><i class="fas fa-save"></i> Schedule Off-Days</button>
                </form>
            </div>
        </div>
    </div>

    <!-- Confirmation Modal -->
    <div class="modal fade" id="confirmModal" tabindex="-1" aria-labelledby="confirmModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="confirmModalLabel">Confirm Off-Scheduling</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    Are you sure you want to schedule off-days for the selected staff?
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="document.getElementById('offScheduleForm').submit()">Confirm</button>
                </div>
            </div>
        </div>
    </div>

    <?php include '../../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
        $(document).ready(function() {
            $('#staff_ids').select2({
                placeholder: 'Select staff',
                allowClear: true
            });

            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
        });
    </script>
</body>
</html>