<?php
session_start();
require_once '../../config/db.php';

// Log script version
error_log("Using roll_call.php version: f8a9b0c1-7d8e-3f4g-9h0i-1j2k3l4m5n6-v7");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Initialize marked_by
$marked_by = null;

// Ensure logs directory exists
$log_dir = __DIR__ . '/../../logs';
$log_file = "$log_dir/roll_call_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}
if (!is_writable($log_dir)) {
    chmod($log_dir, 0755);
}

// Get filters
$branch_id = isset($_GET['branch_id']) ? (int)$_GET['branch_id'] : 0;
$date = $_GET['date'] ?? date('Y-m-d');

// Handle bulk actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_action'])) {
    $staff_ids = $_POST['staff_ids'] ?? [];
    $status = $_POST['bulk_action'];
    
    if (empty($staff_ids)) {
        $_SESSION['error'] = 'No staff selected.';
        header("Location: roll_call.php?date=" . urlencode($date) . "&branch_id=$branch_id");
        exit;
    }

    $conn->begin_transaction();
    $affected = 0;
    try {
        foreach ($staff_ids as $staff_id) {
            $query = "
                INSERT INTO attendance (staff_id, branch_id, attendance_date, status, marked_by, marked_at)
                SELECT staff_id, branch_id, ?, ?, ?, NOW()
                FROM staff WHERE staff_id = ?
                ON DUPLICATE KEY UPDATE status = ?, marked_by = ?, marked_at = NOW()
            ";
            $stmt = $conn->prepare($query);
            $stmt->bind_param('ssissi', $date, $status, $marked_by, $staff_id, $status, $marked_by);
            $stmt->execute();
            $affected += $stmt->affected_rows;
        }
        $conn->commit();
        error_log("[$timestamp] Bulk marked $affected staff as $status by user $marked_by");
        file_put_contents($log_file, "[$timestamp] Bulk marked $affected staff as $status by user $marked_by\n", FILE_APPEND);
        $_SESSION['success'] = "Marked $affected staff as $status.";
    } catch (Exception $e) {
        $conn->rollback();
        error_log("[$timestamp] Bulk update failed: " . $e->getMessage());
        file_put_contents($log_file, "[$timestamp] Bulk update failed: " . $e->getMessage() . "\n", FILE_APPEND);
        $_SESSION['error'] = 'Failed to update roll call: ' . $e->getMessage();
    }
    header("Location: roll_call.php?date=" . urlencode($date) . "&branch_id=$branch_id");
    exit;
}

// Handle individual submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['bulk_action'])) {
    $attendance = $_POST['attendance'] ?? [];
    
    if (empty($attendance)) {
        $_SESSION['error'] = 'No attendance data submitted.';
        header("Location: roll_call.php?date=" . urlencode($date) . "&branch_id=$branch_id");
        exit;
    }

    $conn->begin_transaction();
    $affected = 0;
    try {
        foreach ($attendance as $staff_id => $status) {
            $query = "
                INSERT INTO attendance (staff_id, branch_id, attendance_date, status, marked_by, marked_at)
                SELECT staff_id, branch_id, ?, ?, ?, NOW()
                FROM staff WHERE staff_id = ?
                ON DUPLICATE KEY UPDATE status = ?, marked_by = ?, marked_at = NOW()
            ";
            $stmt = $conn->prepare($query);
            $stmt->bind_param('ssissi', $date, $status, $marked_by, $staff_id, $status, $marked_by);
            $stmt->execute();
            $affected += $stmt->affected_rows;
        }
        $conn->commit();
        error_log("[$timestamp] Roll call updated for $affected staff by user $marked_by");
        file_put_contents($log_file, "[$timestamp] Roll call updated for $affected staff by user $marked_by\n", FILE_APPEND);
        $_SESSION['success'] = "Roll call updated for $affected staff.";
    } catch (Exception $e) {
        $conn->rollback();
        error_log("[$timestamp] Roll call failed: " . $e->getMessage());
        file_put_contents($log_file, "[$timestamp] Roll call failed: " . $e->getMessage() . "\n", FILE_APPEND);
        $_SESSION['error'] = 'Failed to save roll call: ' . $e->getMessage();
    }
    header("Location: roll_call.php?date=" . urlencode($date) . "&branch_id=$branch_id");
    exit;
}

// Fetch branches and staff
$branches = [];
$staff = [];
try {
    $query = "SELECT id, branch_name FROM branches ORDER BY branch_name";
    $result = $conn->query($query);
    while ($row = $result->fetch_assoc()) {
        $branches[] = $row;
    }

    $query = "
        SELECT s.staff_id, s.first_name, s.last_name, sch.shift_type, att.status
        FROM staff s
        LEFT JOIN schedules sch ON s.staff_id = sch.staff_id AND sch.schedule_date = ?
        LEFT JOIN attendance att ON s.staff_id = att.staff_id AND att.attendance_date = ?
        WHERE (? = 0 OR s.branch_id = ?)
        AND (sch.shift_type IS NULL OR sch.shift_type != 'off')
        ORDER BY s.first_name, s.last_name
    ";
    $stmt = $conn->prepare($query);
    $stmt->bind_param('ssii', $date, $date, $branch_id, $branch_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $staff[] = $row;
    }
    $log_marked_by = $marked_by ?? 'unknown';
    error_log("[$timestamp] Fetched " . count($staff) . " staff for roll call by user $log_marked_by");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($staff) . " staff for roll call by user $log_marked_by\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch data failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch data failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load data.';
}

// Close connection
$conn->close();

// Include layout
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Roll Call - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .card { border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .table { font-size: 0.9rem; }
        .status-absent { background-color: #f8d7da; color: #721c24; }
        .status-present { background-color: #d4edda; color: #155724; }
        .btn-primary { background-color: #007bff; border-color: #007bff; }
        .btn-primary:hover { background-color: #0056b3; border-color: #0056b3; }
        .btn-success { background-color: #28a745; border-color: #28a745; }
        .btn-success:hover { background-color: #218838; border-color: #218838; }
        .btn-danger { background-color: #dc3545; border-color: #dc3545; }
        .btn-danger:hover { background-color: #b02a37; border-color: #b02a37; }
        .form-check-input:checked { background-color: #007bff; border-color: #007bff; }
        .alert { border-radius: 8px; }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="mb-4">Roll Call</h2>
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <div class="card mb-4">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-5">
                        <label for="branch_id" class="form-label">Branch</label>
                        <select name="branch_id" id="branch_id" class="form-select">
                            <option value="0">All Branches</option>
                            <?php foreach ($branches as $branch): ?>
                                <option value="<?php echo $branch['id']; ?>" <?php echo $branch_id == $branch['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($branch['branch_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label for="date" class="form-label">Date</label>
                        <input type="date" name="date" id="date" class="form-control" value="<?php echo htmlspecialchars($date); ?>">
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary w-100">Filter</button>
                    </div>
                </form>
            </div>
        </div>
        <div class="card">
            <div class="card-body">
                <form method="POST" id="rollCallForm">
                    <div class="mb-3">
                        <button type="submit" name="bulk_action" value="present" class="btn btn-success" onclick="return checkSelection('bulkModal', 'present')"><i class="fas fa-user-check"></i> Mark Selected Present</button>
                        <button type="submit" name="bulk_action" value="absent" class="btn btn-danger" onclick="return checkSelection('bulkModal', 'absent')"><i class="fas fa-user-times"></i> Mark Selected Absent</button>
                        <button type="submit" class="btn btn-primary"><i class="fas fa-check"></i> Submit Roll Call</button>
                    </div>
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th><input type="checkbox" id="selectAll"></th>
                                <th>Staff ID</th>
                                <th>Name</th>
                                <th>Shift</th>
                                <th>Present</th>
                                <th>Absent</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($staff as $row): ?>
                                <tr <?php echo $row['status'] === 'absent' ? 'class="status-absent"' : ($row['status'] === 'present' ? 'class="status-present"' : ''); ?>>
                                    <td><input type="checkbox" name="staff_ids[]" value="<?php echo $row['staff_id']; ?>" class="select-staff"></td>
                                    <td><?php echo htmlspecialchars($row['staff_id']); ?></td>
                                    <td><?php echo htmlspecialchars($row['first_name'] . ' ' . $row['last_name']); ?></td>
                                    <td><?php echo htmlspecialchars($row['shift_type'] ?? 'N/A'); ?></td>
                                    <td>
                                        <input type="radio" name="attendance[<?php echo $row['staff_id']; ?>]" value="present" <?php echo ($row['status'] === 'present') ? 'checked' : ''; ?> required>
                                    </td>
                                    <td>
                                        <input type="radio" name="attendance[<?php echo $row['staff_id']; ?>]" value="absent" <?php echo ($row['status'] === 'absent') ? 'checked' : ''; ?>>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </form>
            </div>
        </div>
    </div>

    <!-- Bulk Action Modal -->
    <div class="modal fade" id="bulkModal" tabindex="-1" aria-labelledby="bulkModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="bulkModalLabel">Confirm Bulk Action</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    Are you sure you want to mark the selected staff as <span id="modalAction"></span> for <?php echo htmlspecialchars($date); ?>?
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="document.getElementById('rollCallForm').submit()">Confirm</button>
                </div>
            </div>
        </div>
    </div>

    <?php include '../../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Select All Checkbox
        document.getElementById('selectAll').addEventListener('change', function() {
            var checkboxes = document.querySelectorAll('.select-staff');
            checkboxes.forEach(cb => cb.checked = this.checked);
        });

        // Check Selection and Set Modal Action
        function checkSelection(modalId, action) {
            var checkboxes = document.querySelectorAll('input[name="staff_ids[]"]:checked');
            if (checkboxes.length === 0) {
                alert('Please select at least one staff member.');
                return false;
            }
            document.getElementById('modalAction').textContent = action;
            var modal = new bootstrap.Modal(document.getElementById(modalId));
            modal.show();
            return false;
        }
    </script>
</body>
</html>