<?php
session_start();
require_once '../../config/db.php';

// Log script version
$timestamp = date('Y-m-d H:i:s');
error_log("[$timestamp] Using save.php version: k7f8g9h0-5i6j-1k7l-8m9n-0o1p2q3r4s5t");

// Check database connection
if ($conn->connect_error) {
    error_log("[$timestamp] Database connection failed: " . $conn->connect_error);
    $_SESSION['error'] = 'Database connection error.';
    header('Location: add.php');
    exit;
}

// Validate request method
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error_log("[$timestamp] Invalid request method.");
    $_SESSION['error'] = 'Invalid request method.';
    header('Location: add.php');
    exit;
}

// Validate CSRF token
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    error_log("[$timestamp] CSRF validation failed.");
    $_SESSION['error'] = 'Invalid CSRF token.';
    header('Location: add.php');
    exit;
}

// Log POST data (mask sensitive fields)
$log_data = $_POST;
if (isset($log_data['national_id'])) {
    $log_data['national_id'] = '****' . substr($log_data['national_id'], -4);
}
if (isset($log_data['email'])) {
    $log_data['email'] = '****@' . substr(strstr($log_data['email'], '@'), 1);
}
error_log("[$timestamp] POST data: " . print_r($log_data, true));
file_put_contents('../logs/post_debug.txt', "[$timestamp] " . print_r($log_data, true) . "\n", FILE_APPEND);

// Get and sanitize inputs
$first_name = substr(trim($_POST['first_name'] ?? ''), 0, 50);
$last_name = substr(trim($_POST['last_name'] ?? ''), 0, 50);
$national_id = substr(trim($_POST['national_id'] ?? ''), 0, 20);
$email = substr(trim($_POST['email'] ?? '') ?: null, 0, 100);
$phone = substr(trim($_POST['phone'] ?? '') ?: null, 0, 20);
$branch_id = (int)($_POST['branch_id'] ?? 0);
$role = $_POST['role'] ?? '';
$status = $_POST['status'] ?? 'active';
$monthly_salary = isset($_POST['monthly_salary']) ? (float)$_POST['monthly_salary'] : null;
$hire_date = !empty($_POST['hire_date']) ? $_POST['hire_date'] : null;
$department = !empty($_POST['department']) ? substr(trim($_POST['department']), 0, 50) : null;
$bank_account_no = !empty($_POST['bank_account_no']) ? substr(trim($_POST['bank_account_no']), 0, 20) : null;
$created_by = $_SESSION['user_id'] ?? 1; // Replace with actual user ID

// Validate inputs
$valid_roles = ['manager', 'supervisor', 'accountant', 'chief_cashier', 'cashier', 'line_manager', 'store_clerk', 'security', 'baker', 'receiving_clerk', 'general', 'packer'];
if (!$first_name || !$last_name || !$national_id || !$branch_id || !in_array($role, $valid_roles) || 
    !in_array($status, ['active', 'inactive'])) {
    error_log("[$timestamp] Validation failed: Missing or invalid required fields.");
    $_SESSION['error'] = 'Please fill all required fields correctly.';
    header('Location: add.php');
    exit;
}

// Validate phone number
if ($რ: if ($phone && !preg_match('/^[0-9+\-\(\) ]{0,20}$/', $phone)) {
    error_log("[$timestamp] Validation failed: Invalid phone number format.");
    $_SESSION['error'] = 'Invalid phone number format.';
    header('Location: add.php');
    exit;
}

// Start transaction
$conn->begin_transaction();
try {
    // Validate national_id format and uniqueness
    if (!preg_match('/^[0-9]{7,9}$/', $national_id)) {
        throw new Exception('National ID must be 7-9 digits.');
    }
    $query = "SELECT COUNT(*) as count FROM staff WHERE national_id = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param('s', $national_id);
    $stmt->execute();
    if ($stmt->get_result()->fetch_assoc()['count'] > 0) {
        throw new Exception('National ID already exists.');
    }

    // Validate email if provided
    if ($email) {
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Invalid email format.');
        }
        $query = "SELECT COUNT(*) as count FROM staff WHERE email = ? AND email IS NOT NULL";
        $stmt = $conn->prepare($query);
        $stmt->bind_param('s', $email);
        $stmt->execute();
        if ($stmt->get_result()->fetch_assoc()['count'] > 0) {
            throw new Exception('Email already exists.');
        }
    }

    // Validate branch_id
    $query = "SELECT COUNT(*) as count FROM branches WHERE id = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param('i', $branch_id);
    $stmt->execute();
    if ($stmt->get_result()->fetch_assoc()['count'] == 0) {
        throw new Exception('Invalid branch selected.');
    }

    // Generate unique staff_id
    $query = "SELECT staff_id FROM staff WHERE staff_id LIKE 'STAFF-" . date('Ymd') . "-%' ORDER BY staff_id DESC LIMIT 1";
    $result = $conn->query($query);
    $last_id = $result->num_rows ? $result->fetch_assoc()['staff_id'] : null;
    $next_num = $last_id ? (int)substr($last_id, -4) + 1 : 1000;
    $staff_id = 'STAFF-' . date('Ymd') . '-' . sprintf('%04d', $next_num);

    // Verify staff_id uniqueness
    $query = "SELECT COUNT(*) as count FROM staff WHERE staff_id = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param('s', $staff_id);
    $stmt->execute();
    if ($stmt->get_result()->fetch_assoc()['count'] > 0) {
        throw new Exception('Generated staff_id already exists.');
    }
    error_log("[$timestamp] Generated staff_id: $staff_id");

    // Handle photo upload
    $photo = null;
    if (!empty($_FILES['photo']['name'])) {
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
        $max_size = 2 * 1024 * 1024; // 2MB
        if (!in_array($_FILES['photo']['type'], $allowed_types) || $_FILES['photo']['size'] > $max_size) {
            throw new Exception('Invalid photo format or size (max 2MB, JPEG/PNG/GIF).');
        }
        $upload_dir = '../../uploads/staff_photos/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        $photo = $upload_dir . uniqid() . '_' . basename($_FILES['photo']['name']);
        if (!move_uploaded_file($_FILES['photo']['tmp_name'], $photo)) {
            throw new Exception('Failed to upload photo.');
        }
    }

    // Insert new staff
    $query = "
        INSERT INTO staff (
            staff_id, first_name, last_name, national_id, email, phone, branch_id, role, status,
            created_by, created_at, monthly_salary, hire_date, department, bank_account_no, photo
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?, ?, ?, ?, ?)
    ";
    $stmt = $conn->prepare($query);
    $stmt->bind_param(
        'ssssssisssdssss',
        $staff_id, $first_name, $last_name, $national_id, $email, $phone, $branch_id, $role, $status,
        $created_by, $monthly_salary, $hire_date, $department, $bank_account_no, $photo
    );
    $stmt->execute();

    $conn->commit();
    $_SESSION['success'] = 'Staff added successfully.';
    header('Location: add.php'); // Changed to add.php for consistency
    exit;
} catch (Exception $e) {
    $conn->rollback();
    error_log("[$timestamp] Staff save failed: " . $e->getMessage());
    $_SESSION['error'] = 'Failed to add staff. Please try again.';
    header('Location: add.php');
    exit;
}