<?php
session_start();
require_once '../../config/db.php';
require_once '../../vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\IOFactory;

// Log script version
error_log("Using staff_payments/add_employees.php version: p4e5f6a7-b8c9-4d0e-f1g2-3h4i5j6k7l8m");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = '../logs';
$log_file = "$log_dir/staff_payments_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Generate CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Handle Excel file upload
$results = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['excel_file']) && isset($_POST['upload_employees'])) {
    if ($_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = 'Invalid CSRF token.';
        header("Location: add_employees.php");
        exit;
    }

    $file = $_FILES['excel_file'];
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $_SESSION['error'] = 'File upload failed: ' . $file['error'];
        header("Location: add_employees.php");
        exit;
    }

    $allowed_extensions = ['xlsx', 'xls'];
    $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($file_ext, $allowed_extensions)) {
        $_SESSION['error'] = 'Invalid file format. Please upload an Excel file (.xlsx or .xls).';
        header("Location: add_employees.php");
        exit;
    }

    try {
        $spreadsheet = IOFactory::load($file['tmp_name']);
        $worksheet = $spreadsheet->getActiveSheet();
        $rows = $worksheet->toArray();
        $headers = array_shift($rows); // Remove header row

        // Expected headers
        $expected_headers = ['Debit Account No', 'Beneficiary Account', 'Beneficiary Name', 'Currency', 'Amount'];
        if (array_slice($headers, 0, 5) !== $expected_headers) {
            $_SESSION['error'] = 'Invalid Excel format. Expected headers: ' . implode(', ', $expected_headers);
            header("Location: add_employees.php");
            exit;
        }

        $conn->begin_transaction();
        $stmt_check = $conn->prepare("SELECT staff_id FROM staff WHERE bank_account_no = ?");
        $stmt_insert = $conn->prepare("
            INSERT INTO staff (staff_id, first_name, last_name, national_id, role, branch_id, bank_account_no, monthly_salary, payment_preference, created_by, created_at, status)
            VALUES (?, ?, ?, ?, 'staff', 1, ?, ?, 'mid-month', ?, NOW(), 'active')
        ");

        foreach ($rows as $index => $row) {
            // Skip empty rows or total row
            if (empty($row[1]) || !is_numeric($row[4])) {
                continue;
            }

            $bank_account_no = $row[1] ?? '';
            $name = $row[2] ?? '';
            $currency = $row[3] ?? '';
            $amount = (float)($row[4] ?? 0);

            // Validate data
            if (empty($bank_account_no) || $currency !== 'KES' || $amount <= 0 || !preg_match('/^[0-9]{10,20}$/', $bank_account_no)) {
                $results[] = [
                    'bank_account_no' => $bank_account_no,
                    'name' => $name,
                    'status' => 'Failed',
                    'reason' => 'Invalid data: Missing bank account, invalid currency, or non-positive amount'
                ];
                continue;
            }

            // Check for duplicate bank account
            $stmt_check->bind_param('s', $bank_account_no);
            $stmt_check->execute();
            if ($stmt_check->get_result()->fetch_assoc()) {
                $results[] = [
                    'bank_account_no' => $bank_account_no,
                    'name' => $name,
                    'status' => 'Failed',
                    'reason' => 'Bank account already exists'
                ];
                continue;
            }

            // Split name
            $name_parts = explode(' ', trim($name));
            $last_name = array_pop($name_parts);
            $first_name = implode(' ', $name_parts);
            if (empty($first_name)) {
                $first_name = $last_name;
                $last_name = '';
            }

            // Calculate minimum monthly_salary (Amount ≤ 40% of salary)
            $monthly_salary = ceil($amount / 0.4);

            // Generate staff_id and national_id
            $staff_id = 'ST' . str_pad($index + 1, 4, '0', STR_PAD_LEFT); // e.g., ST0001
            $national_id = sprintf('%08d', rand(10000000, 99999999)); // Random 8-digit

            // Insert employee
            $stmt_insert->bind_param('ssssds', $staff_id, $first_name, $last_name, $national_id, $bank_account_no, $monthly_salary, $created_by);
            $stmt_insert->execute();

            $results[] = [
                'bank_account_no' => $bank_account_no,
                'name' => $name,
                'staff_id' => $staff_id,
                'monthly_salary' => $monthly_salary,
                'status' => 'Success',
                'reason' => ''
            ];

            error_log("[$timestamp] Added employee $staff_id ($name) with bank account $bank_account_no");
            file_put_contents($log_file, "[$timestamp] Added employee $staff_id ($name) with bank account $bank_account_no\n", FILE_APPEND);
        }

        $conn->commit();

        // Generate CSV report
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="employee_import_results_' . date('YmdHis') . '.csv"');
        $output = fopen('php://output', 'w');
        fputcsv($output, ['Staff ID', 'Name', 'Bank Account No', 'Monthly Salary (KES)', 'Status', 'Reason']);
        foreach ($results as $result) {
            fputcsv($output, [
                $result['staff_id'] ?? 'N/A',
                $result['name'],
                $result['bank_account_no'],
                number_format($result['monthly_salary'] ?? 0, 2),
                $result['status'],
                $result['reason']
            ]);
        }
        fclose($output);
        exit;
    } catch (Exception $e) {
        $conn->rollback();
        $log_message = "[$timestamp] Employee import failed: " . $e->getMessage() . " | Created By: $created_by";
        error_log($log_message);
        file_put_contents($log_file, "$log_message\n", FILE_APPEND);
        $_SESSION['error'] = 'Failed to import employees: ' . $e->getMessage();
        header("Location: add_employees.php");
        exit;
    }
}

// Close connection
$conn->close();

// Include layout
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Employees - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); border: 1px solid #e0e0e0; }
        .btn-primary { background-color: #007bff; border-color: #007bff; transition: all 0.3s ease; }
        .btn-primary:hover { background-color: #0056b3; border-color: #0056b3; box-shadow: 0 2px 8px rgba(0,0,0,0.2); }
        .form-label { font-weight: 500; color: #333; margin-bottom: 8px; }
        .form-control { border: 1px solid #007bff; border-radius: 5px; transition: border-color 0.3s ease; }
        .form-control:hover { border-color: #0056b3; }
        .alert { border-radius: 8px; }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="mb-4">Add Employees from Excel</h2>
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <div class="card">
            <div class="card-body">
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                    <div class="mb-3">
                        <label for="excel_file" class="form-label">Select Employee Excel File (.xlsx, .xls)</label>
                        <input type="file" name="excel_file" id="excel_file" class="form-control" accept=".xlsx,.xls" required>
                    </div>
                    <button type="submit" name="upload_employees" class="btn btn-primary"><i class="fas fa-upload"></i> Import Employees</button>
                    <a href="list.php" class="btn btn-secondary">Cancel</a>
                </form>
            </div>
        </div>
    </div>

    <?php include '../../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>