<?php
session_start();
require_once '../../config/db.php';

// Log script version
error_log("Using staff_payments/bulk_pay.php version: n2c3d4e5-f6a7-2b8c-d9e0-1f2g3h4i5j6k");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = '../logs';
$log_file = "$log_dir/staff_payments_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Generate CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Fetch active staff
$staff_list = [];
try {
    $stmt = $conn->prepare("
        SELECT s.staff_id, s.first_name, s.last_name, s.monthly_salary, s.bank_account_no,
               (SELECT SUM(sp.amount) FROM staff_payments sp WHERE sp.staff_id = s.staff_id AND sp.payment_date LIKE ? AND sp.payment_type IN ('monthly', 'mid-month')) AS paid_this_month,
               (SELECT SUM(l.monthly_deduction) FROM loans l WHERE l.staff_id = s.staff_id AND l.status = 'approved' AND l.monthly_deduction > 0) AS loan_deduction
        FROM staff s
        WHERE s.status = 'active'
        ORDER BY s.first_name, s.last_name
    ");
    $month = date('Y-m');
    $month_like = "$month%";
    $stmt->bind_param('s', $month_like);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $row['remaining_payable'] = $row['monthly_salary'] - ($row['paid_this_month'] ?? 0) - ($row['loan_deduction'] ?? 0);
        $row['advance_amount'] = $row['monthly_salary'] * 0.4; // 40% for mid-month advance
        $staff_list[] = $row;
    }
    error_log("[$timestamp] Fetched " . count($staff_list) . " active staff");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($staff_list) . " active staff\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch staff failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch staff failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load staff list.';
    header("Location: list.php");
    exit;
}

// Handle payment submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_payments'])) {
    if ($_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = 'Invalid CSRF token.';
        header("Location: bulk_pay.php");
        exit;
    }

    $payment_mode = $_POST['payment_mode'] ?? 'monthly';
    $selected_staff = $_POST['selected_staff'] ?? [];
    $amounts = $_POST['amount'] ?? [];
    $payment_types = $_POST['payment_type'] ?? [];
    $payment_dates = $_POST['payment_date'] ?? [];
    $bank_accounts = $_POST['bank_account_no'] ?? [];
    $payment_results = [];
    $total_amount = 0;
    $total_employees = 0;

    if (empty($selected_staff)) {
        $_SESSION['error'] = 'No staff selected for payment.';
        header("Location: bulk_pay.php");
        exit;
    }

    // Revalidate created_by
    $created_by = $_SESSION['user_id'] ?? 'ADMIN';
    try {
        $stmt = $conn->prepare("SELECT staff_id FROM staff WHERE staff_id = ? AND status = 'active'");
        $stmt->bind_param('s', $created_by);
        $stmt->execute();
        $result = $stmt->get_result();
        if (!$result->fetch_assoc()) {
            error_log("[$timestamp] Revalidation failed for user_id: $created_by");
            file_put_contents($log_file, "[$timestamp] Revalidation failed for user_id: $created_by\n", FILE_APPEND);
            $created_by = 'ADMIN';
            $stmt->bind_param('s', $created_by);
            $stmt->execute();
            $result = $stmt->get_result();
            if (!$result->fetch_assoc()) {
                $created_by = !empty($staff_list) ? $staff_list[0]['staff_id'] : null;
                if ($created_by) {
                    $stmt->bind_param('s', $created_by);
                    $stmt->execute();
                    if (!$stmt->get_result()->fetch_assoc()) {
                        $_SESSION['error'] = 'No valid staff available for processing.';
                        header("Location: bulk_pay.php");
                        exit;
                    }
                } else {
                    $_SESSION['error'] = 'No active staff available.';
                    header("Location: bulk_pay.php");
                    exit;
                }
            }
            error_log("[$timestamp] Fallback to created_by: $created_by");
            file_put_contents($log_file, "[$timestamp] Fallback to created_by: $created_by\n", FILE_APPEND);
        }
    } catch (Exception $e) {
        error_log("[$timestamp] Revalidate user failed: " . $e->getMessage());
        file_put_contents($log_file, "[$timestamp] Revalidate user failed: " . $e->getMessage() . "\n", FILE_APPEND);
        $_SESSION['error'] = 'Failed to revalidate user.';
        header("Location: bulk_pay.php");
        exit;
    }

    $conn->begin_transaction();
    try {
        $stmt_payment = $conn->prepare("
            INSERT INTO staff_payments (staff_id, branch_id, amount, payment_type, payment_date, bank_account_no, reference_no, created_by)
            SELECT ?, branch_id, ?, ?, ?, ?, ?, ? FROM staff WHERE staff_id = ?
        ");
        $stmt_loan = $conn->prepare("
            INSERT INTO staff_payments (staff_id, branch_id, amount, payment_type, payment_date, bank_account_no, reference_no, created_by)
            SELECT ?, branch_id, ?, 'loan_repayment', ?, ?, ?, ? FROM staff WHERE staff_id = ?
        ");

        $batch_size = 20;
        $batches = array_chunk($selected_staff, $batch_size);
        foreach ($batches as $batch) {
            foreach ($batch as $staff_id) {
                if (!isset($amounts[$staff_id], $payment_types[$staff_id], $payment_dates[$staff_id], $bank_accounts[$staff_id])) {
                    $payment_results[] = [
                        'staff_id' => $staff_id,
                        'name' => 'Unknown',
                        'amount' => 0,
                        'bank_account_no' => $bank_accounts[$staff_id] ?? '',
                        'status' => 'Failed',
                        'reason' => 'Missing payment details'
                    ];
                    continue;
                }

                $amount = (float)$amounts[$staff_id];
                $payment_type = $payment_types[$staff_id];
                $payment_date = $payment_dates[$staff_id];
                $bank_account_no = $bank_accounts[$staff_id];

                // Find staff data
                $staff = null;
                foreach ($staff_list as $s) {
                    if ($s['staff_id'] === $staff_id) {
                        $staff = $s;
                        break;
                    }
                }

                if (!$staff) {
                    $payment_results[] = [
                        'staff_id' => $staff_id,
                        'name' => 'Unknown',
                        'amount' => $amount,
                        'bank_account_no' => $bank_account_no,
                        'status' => 'Failed',
                        'reason' => 'Staff not found'
                    ];
                    continue;
                }

                // Validate inputs
                if ($amount <= 0 || !in_array($payment_type, ['monthly', 'mid-month']) || empty($payment_date) || empty($bank_account_no) || !preg_match('/^[0-9]{10,20}$/', $bank_account_no)) {
                    $payment_results[] = [
                        'staff_id' => $staff_id,
                        'name' => $staff['first_name'] . ' ' . $staff['last_name'],
                        'amount' => $amount,
                        'bank_account_no' => $bank_account_no,
                        'status' => 'Failed',
                        'reason' => 'Invalid payment details'
                    ];
                    continue;
                }

                // Check amount limits
                $max_allowed = $payment_type === 'mid-month' ? $staff['advance_amount'] : $staff['remaining_payable'];
                if ($amount > $max_allowed) {
                    $payment_results[] = [
                        'staff_id' => $staff_id,
                        'name' => $staff['first_name'] . ' ' . $staff['last_name'],
                        'amount' => $amount,
                        'bank_account_no' => $bank_account_no,
                        'status' => 'Failed',
                        'reason' => 'Amount exceeds ' . ($payment_type === 'mid-month' ? 'advance limit' : 'remaining salary') . ' (KSh ' . number_format($max_allowed, 2) . ')'
                    ];
                    continue;
                }

                // Generate unique reference number
                $reference_no = 'PAY-' . str_replace(['-', ' ', ':'], '', $timestamp) . '-' . rand(1000, 9999);

                // Insert payment
                $stmt_payment->bind_param('sdssssss', $staff_id, $amount, $payment_type, $payment_date, $bank_account_no, $reference_no, $created_by, $staff_id);
                $stmt_payment->execute();

                // Deduct loan if applicable (only for monthly payments)
                if ($staff['loan_deduction'] > 0 && $payment_type === 'monthly') {
                    $loan_ref = 'LOAN-' . $reference_no;
                    $stmt_loan->bind_param('sdssssss', $staff_id, $staff['loan_deduction'], $payment_date, $bank_account_no, $loan_ref, $created_by, $staff_id);
                    $stmt_loan->execute();
                }

                // Add to payment results for CSV
                $payment_results[] = [
                    'debit_account_no' => '0920296445438',
                    'beneficiary_account' => $bank_account_no,
                    'beneficiary_name' => strtoupper($staff['first_name'] . ' ' . $staff['last_name']),
                    'currency' => 'KES',
                    'amount' => $amount
                ];

                // Update totals
                $total_amount += $amount;
                $total_employees++;

                error_log("[$timestamp] Payment of KSh $amount ($payment_type) for staff ID $staff_id by $created_by");
                file_put_contents($log_file, "[$timestamp] Payment of KSh $amount ($payment_type) for staff ID $staff_id by $created_by\n", FILE_APPEND);
            }
        }

        $conn->commit();

        // Store totals in session for page display
        $_SESSION['payment_summary'] = [
            'total_amount' => $total_amount,
            'total_employees' => $total_employees
        ];

        // Generate CSV
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="payment_results_' . date('YmdHis') . '.csv"');
        $output = fopen('php://output', 'w');
        fputcsv($output, ['Debit Account No', 'Beneficiary Account', 'Beneficiary Name', 'Currency', 'Amount']);
        foreach ($payment_results as $result) {
            fputcsv($output, [
                $result['debit_account_no'],
                $result['beneficiary_account'],
                $result['beneficiary_name'],
                $result['currency'],
                number_format($result['amount'], 0)
            ]);
        }
        // Add totals row
        fputcsv($output, ['', '', 'Total Employees: ' . $total_employees, 'Total Amount: KES', number_format($total_amount, 0)]);
        fclose($output);
        exit;
    } catch (Exception $e) {
        $conn->rollback();
        $log_message = "[$timestamp] Bulk payment failed: " . $e->getMessage() . " | Created By: $created_by";
        error_log($log_message);
        file_put_contents($log_file, "$log_message\n", FILE_APPEND);
        $_SESSION['error'] = 'Failed to process bulk payments: ' . $e->getMessage();
        header("Location: bulk_pay.php");
        exit;
    }
}

// Close connection
$conn->close();

// Include layout
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bulk Payment Processing - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); border: 1px solid #e0e0e0; }
        .btn-primary { background-color: #007bff; border-color: #007bff; transition: all 0.3s ease; }
        .btn-primary:hover { background-color: #0056b3; border-color: #0056b3; box-shadow: 0 2px 8px rgba(0,0,0,0.2); }
        .form-label { font-weight: 500; color: #333; margin-bottom: 8px; }
        .form-control, .form-select { border: 1px solid #007bff; border-radius: 5px; transition: border-color 0.3s ease; }
        .form-control:hover, .form-select:hover { border-color: #0056b3; }
        .alert { border-radius: 8px; }
        .table { font-size: 0.9rem; }
        .table-responsive { max-height: 500px; overflow-y: auto; }
        .apply-btn { background: linear-gradient(to right, #28a745, #20c997); border: none; border-radius: 5px; padding: 10px 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.2); transition: all 0.3s ease; }
        .apply-btn:hover { background: linear-gradient(to right, #218838, #1c8b73); box-shadow: 0 4px 12px rgba(0,0,0,0.3); }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="mb-4">Bulk Payment Processing</h2>
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (isset($_SESSION['payment_summary'])): ?>
            <div class="alert alert-info alert-dismissible fade show" role="alert">
                Processed <strong><?php echo $_SESSION['payment_summary']['total_employees']; ?></strong> employees for a total of <strong>KSh <?php echo number_format($_SESSION['payment_summary']['total_amount'], 2); ?></strong>.
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['payment_summary']); ?>
        <?php endif; ?>
        <div class="card">
            <div class="card-body">
                <form method="POST" id="bulkPaymentForm">
                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                    <div class="mb-3">
                        <label for="payment_mode" class="form-label">Payment Mode</label>
                        <select name="payment_mode" id="payment_mode" class="form-select" onchange="applyDefaults()">
                            <option value="monthly">End-Month Salary (60%)</option>
                            <option value="mid-month">Mid-Month Advance (40%)</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <button type="button" class="btn apply-btn" onclick="applyDefaults()"><i class="fas fa-check-circle"></i> Apply Defaults</button>
                    </div>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="table-dark">
                                <tr>
                                    <th><input type="checkbox" id="selectAll" onchange="toggleSelectAll()"></th>
                                    <th>Staff ID</th>
                                    <th>Name</th>
                                    <th>Monthly Salary (KES)</th>
                                    <th>Paid This Month (KES)</th>
                                    <th>Loan Deduction (KES)</th>
                                    <th>Remaining Payable (KES)</th>
                                    <th>Advance Limit (KES)</th>
                                    <th>Amount (KES)</th>
                                    <th>Payment Type</th>
                                    <th>Payment Date</th>
                                    <th>Bank Account No</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($staff_list as $staff): ?>
                                    <tr>
                                        <td><input type="checkbox" name="selected_staff[]" value="<?php echo htmlspecialchars($staff['staff_id']); ?>" class="staff-checkbox"></td>
                                        <td><?php echo htmlspecialchars($staff['staff_id']); ?></td>
                                        <td><?php echo htmlspecialchars($staff['first_name'] . ' ' . $staff['last_name']); ?></td>
                                        <td>KSh <?php echo number_format($staff['monthly_salary'], 2); ?></td>
                                        <td>KSh <?php echo number_format($staff['paid_this_month'] ?? 0, 2); ?></td>
                                        <td>KSh <?php echo number_format($staff['loan_deduction'] ?? 0, 2); ?></td>
                                        <td>KSh <?php echo number_format($staff['remaining_payable'], 2); ?></td>
                                        <td>KSh <?php echo number_format($staff['advance_amount'], 2); ?></td>
                                        <td>
                                            <input type="number" step="0.01" name="amount[<?php echo htmlspecialchars($staff['staff_id']); ?>]" class="form-control amount" value="<?php echo number_format($staff['remaining_payable'], 2); ?>">
                                        </td>
                                        <td>
                                            <select name="payment_type[<?php echo htmlspecialchars($staff['staff_id']); ?>]" class="form-select payment-type">
                                                <option value="monthly" selected>Monthly Salary</option>
                                                <option value="mid-month">Mid-Month Advance</option>
                                            </select>
                                        </td>
                                        <td>
                                            <input type="date" name="payment_date[<?php echo htmlspecialchars($staff['staff_id']); ?>]" class="form-control payment-date" value="<?php echo date('Y-m-d'); ?>">
                                        </td>
                                        <td>
                                            <input type="text" name="bank_account_no[<?php echo htmlspecialchars($staff['staff_id']); ?>]" class="form-control bank-account" value="<?php echo htmlspecialchars($staff['bank_account_no']); ?>">
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <button type="submit" name="submit_payments" class="btn btn-primary mt-3"><i class="fas fa-money-bill"></i> Process Payments</button>
                    <a href="list.php" class="btn btn-secondary mt-3">Cancel</a>
                </form>
            </div>
        </div>
    </div>

    <?php include '../../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function toggleSelectAll() {
            const selectAll = document.getElementById('selectAll');
            const checkboxes = document.querySelectorAll('.staff-checkbox');
            checkboxes.forEach(cb => cb.checked = selectAll.checked);
        }

        function applyDefaults() {
            const paymentMode = document.getElementById('payment_mode').value;
            const amounts = document.querySelectorAll('.amount');
            const paymentTypes = document.querySelectorAll('.payment-type');
            const paymentDates = document.querySelectorAll('.payment-date');
            const today = '<?php echo date('Y-m-d'); ?>';
            amounts.forEach((input, index) => {
                const row = input.closest('tr');
                const remaining = parseFloat(row.cells[6].textContent.replace('KSh ', '').replace(/,/g, '')) || 0;
                const advance = parseFloat(row.cells[7].textContent.replace('KSh ', '').replace(/,/g, '')) || 0;
                input.value = (paymentMode === 'mid-month' ? advance : remaining).toFixed(2);
                paymentTypes[index].value = paymentMode;
                paymentDates[index].value = today;
            });
        }

        // Auto-apply defaults on page load
        document.addEventListener('DOMContentLoaded', applyDefaults);
    </script>
</body>
</html>