<?php
session_start();
require_once '../../config/db.php';

// Log script version
error_log("Using staff_payments/list.php version: f7b8c9d0-e1f2-4a3b-9c4d-5e6f7a8b9c0d");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = '../logs';
$log_file = "$log_dir/staff_payments_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Generate CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Get filters
$branch_id = isset($_GET['branch_id']) ? (int)$_GET['branch_id'] : 0;
$role = $_GET['role'] ?? '';
$month = $_GET['month'] ?? date('Y-m');

// Fetch branches
$branches = [];
try {
    $result = $conn->query("SELECT id, branch_name FROM branches ORDER BY branch_name");
    $branches = $result->fetch_all(MYSQLI_ASSOC);
    error_log("[$timestamp] Fetched " . count($branches) . " branches");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($branches) . " branches\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch branches failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch branches failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load branches.';
}

// Fetch roles
$roles = ['manager', 'supervisor', 'accountant', 'chief_cashier', 'cashier', 'line_manager', 'store_clerk', 'security', 'baker', 'receiving_clerk', 'general', 'packer'];

// Fetch staff payment data
$staff = [];
$total_paid = 0;
try {
    $query = "
        SELECT s.staff_id, s.first_name, s.last_name, s.national_id, s.role, s.monthly_salary, s.payment_preference, b.branch_name,
               (SELECT SUM(sp.amount) FROM staff_payments sp WHERE sp.staff_id = s.staff_id AND sp.payment_date LIKE ? AND sp.payment_type IN ('monthly', 'mid-month')) AS paid_this_month,
               (SELECT SUM(l.monthly_deduction) FROM loans l WHERE l.staff_id = s.staff_id AND l.status = 'approved' AND l.monthly_deduction > 0) AS loan_deduction
        FROM staff s
        JOIN branches b ON s.branch_id = b.id
        WHERE s.status = 'active' AND (? = 0 OR s.branch_id = ?)
        AND (? = '' OR s.role = ?)
        ORDER BY s.first_name, s.last_name
    ";
    $month_like = "$month%";
    $stmt = $conn->prepare($query);
    $stmt->bind_param('siiss', $month_like, $branch_id, $branch_id, $role, $role);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $staff[] = $row;
        $total_paid += $row['paid_this_month'] ?? 0;
    }
    error_log("[$timestamp] Fetched " . count($staff) . " staff");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($staff) . " staff\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch staff failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch staff failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load staff list.';
}

// Close connection
$conn->close();

// Include layout
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Staff Payments - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet">
    <style>
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); border: 1px solid #e0e0e0; }
        .filter-card { background: linear-gradient(to bottom, #007bff, #0056b3); color: white; }
        .filter-card .card-body { background: white; border-radius: 0 0 10px 10px; }
        .table { font-size: 0.9rem; }
        .status-paid { background-color: #d4edda; color: #155724; }
        .status-pending { background-color: #fff3cd; color: #856404; }
        .btn-primary { background-color: #007bff; border-color: #007bff; transition: all 0.3s ease; }
        .btn-primary:hover { background-color: #0056b3; border-color: #0056b3; box-shadow: 0 2px 8px rgba(0,0,0,0.2); }
        .btn-success { background-color: #28a745; border-color: #28a745; }
        .btn-success:hover { background-color: #218838; border-color: #218838; }
        .alert { border-radius: 8px; }
        .select2-container { width: 100% !important; }
        .select2-container .select2-selection--single {
            border: 1px solid #007bff; border-radius: 5px; height: 38px; padding: 5px; transition: border-color 0.3s ease;
        }
        .select2-container .select2-selection--single:hover { border-color: #0056b3; }
        .select2-container .select2-selection__rendered { color: #333; line-height: 28px; }
        .select2-container .select2-selection__arrow { height: 36px; }
        .select2-dropdown { border: 1px solid #007bff; border-radius: 5px; animation: fadeIn 0.2s ease-in; }
        .month-input { border: 1px solid #007bff; border-radius: 5px; transition: border-color 0.3s ease; }
        .month-input:hover { border-color: #0056b3; }
        .form-label { font-weight: 500; color: #333; margin-bottom: 8px; }
        .tooltip-inner { background-color: #333; border-radius: 5px; padding: 8px; }
        .tooltip.bs-tooltip-top .tooltip-arrow::before { border-top-color: #333; }
        .filter-btn { background: linear-gradient(to right, #007bff, #00aaff); border: none; border-radius: 5px; padding: 10px 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.2); transition: all 0.3s ease; }
        .filter-btn:hover { background: linear-gradient(to right, #0056b3, #0088cc); box-shadow: 0 4px 12px rgba(0,0,0,0.3); }
        .table-responsive { max-height: 500px; overflow-y: auto; }
        .table tfoot tr { position: sticky; bottom: 0; z-index: 1; background-color: #343a40; color: white; }
        @keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }
        @media (max-width: 576px) {
            .filter-card .row > div { margin-bottom: 15px; }
            .filter-btn { width: 100%; }
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="mb-4">Staff Payments Portal</h2>
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <div class="card filter-card mb-4">
            <div class="card-body p-4">
                <form method="GET" class="row g-3">
                    <div class="col-md-4">
                        <label for="branch_id" class="form-label">Branch <i class="fas fa-info-circle" data-bs-toggle="tooltip" title="Select a branch to filter staff"></i></label>
                        <select name="branch_id" id="branch_id" class="form-select" aria-label="Select branch">
                            <option value="0">All Branches</option>
                            <?php foreach ($branches as $branch): ?>
                                <option value="<?php echo $branch['id']; ?>" <?php echo $branch_id == $branch['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($branch['branch_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label for="role" class="form-label">Role <i class="fas fa-info-circle" data-bs-toggle="tooltip" title="Select a role to filter staff"></i></label>
                        <select name="role" id="role" class="form-select" aria-label="Select role">
                            <option value="">All Roles</option>
                            <?php foreach ($roles as $r): ?>
                                <option value="<?php echo $r; ?>" <?php echo $role == $r ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($r); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label for="month" class="form-label">Month <i class="fas fa-info-circle" data-bs-toggle="tooltip" title="Select a month to view payments"></i></label>
                        <input type="month" name="month" id="month" class="form-control month-input" value="<?php echo htmlspecialchars($month); ?>" aria-label="Select month">
                    </div>
                    <div class="col-12 d-flex justify-content-end">
                        <button type="submit" class="btn filter-btn"><i class="fas fa-filter"></i> Apply Filters</button>
                    </div>
                </form>
            </div>
        </div>
        <div class="card">
            <div class="card-body table-responsive">
                <table class="table table-striped table-hover">
                    <thead class="table-dark">
                        <tr>
                            <th>Name</th>
                            <th>Staff ID</th>
                            <th>National ID</th>
                            <th>Role</th>
                            <th>Branch</th>
                            <th>Monthly Salary (KES)</th>
                            <th>Paid This Month (KES)</th>
                            <th>Loan Deduction (KES)</th>
                            <th>Payment Preference</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($staff as $row): ?>
                            <?php
                            $paid = $row['paid_this_month'] ?? 0;
                            $salary = $row['monthly_salary'] ?? 0;
                            $status = ($paid >= $salary && $salary > 0) ? 'Paid' : 'Pending';
                            ?>
                            <tr class="<?php echo $status === 'Paid' ? 'status-paid' : 'status-pending'; ?>">
                                <td><?php echo htmlspecialchars($row['first_name'] . ' ' . $row['last_name']); ?></td>
                                <td><?php echo htmlspecialchars($row['staff_id']); ?></td>
                                <td><?php echo htmlspecialchars($row['national_id']); ?></td>
                                <td><?php echo htmlspecialchars($row['role']); ?></td>
                                <td><?php echo htmlspecialchars($row['branch_name']); ?></td>
                                <td class="text-end">KSh <?php echo number_format($row['monthly_salary'], 2); ?></td>
                                <td class="text-end">KSh <?php echo number_format($row['paid_this_month'] ?? 0, 2); ?></td>
                                <td class="text-end">KSh <?php echo number_format($row['loan_deduction'] ?? 0, 2); ?></td>
                                <td><?php echo htmlspecialchars($row['payment_preference']); ?></td>
                                <td><?php echo $status; ?></td>
                                <td>
                                    <a href="pay.php?staff_id=<?php echo urlencode($row['staff_id']); ?>" class="btn btn-sm btn-success"><i class="fas fa-money-bill"></i> Pay</a>
                                    <a href="loan.php?staff_id=<?php echo urlencode($row['staff_id']); ?>" class="btn btn-sm btn-warning"><i class="fas fa-hand-holding-usd"></i> Loan</a>
                                    <a href="view.php?staff_id=<?php echo urlencode($row['staff_id']); ?>" class="btn btn-sm btn-info"><i class="fas fa-eye"></i> View</a>
                                    <a href="bio_data.php?staff_id=<?php echo urlencode($row['staff_id']); ?>" class="btn btn-sm btn-primary"><i class="fas fa-user-edit"></i> Bio-Data</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot>
                        <tr>
                            <td colspan="6" class="text-end"><strong>Total Paid</strong></td>
                            <td class="text-end"><strong>KSh <?php echo number_format($total_paid, 2); ?></strong></td>
                            <td colspan="4"></td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
    </div>

    <?php include '../../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
        $(document).ready(function() {
            $('#branch_id, #role').select2({
                placeholder: 'Select an option',
                allowClear: true,
                dropdownCssClass: 'select2-custom-dropdown'
            });

            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl, {
                    animation: true,
                    delay: { show: 100, hide: 200 }
                });
            });
        });
    </script>
</body>
</html>