<?php
session_start();
require_once '../../config/db.php';

// Log script version
error_log("Using staff_payments/pay.php version: d2e3f4a5-b6c7-4d8e-9f0a-1b2c3d4e5f6b");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = '../logs';
$log_file = "$log_dir/staff_payments_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Generate CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Get staff_id
$staff_id = $_GET['staff_id'] ?? '';
if (empty($staff_id)) {
    $_SESSION['error'] = 'Invalid staff ID.';
    header("Location: list.php");
    exit;
}

// Fetch staff details
try {
    $stmt = $conn->prepare("
        SELECT s.staff_id, s.first_name, s.last_name, s.monthly_salary, s.bank_account_no, s.payment_preference,
               (SELECT SUM(sp.amount) FROM staff_payments sp WHERE sp.staff_id = s.staff_id AND sp.payment_date LIKE ? AND sp.payment_type IN ('monthly', 'mid-month')) AS paid_this_month,
               (SELECT SUM(l.monthly_deduction) FROM loans l WHERE l.staff_id = s.staff_id AND l.status = 'approved' AND l.monthly_deduction > 0) AS loan_deduction
        FROM staff s
        WHERE s.staff_id = ? AND s.status = 'active'
    ");
    $month = date('Y-m');
    $month_like = "$month%";
    $stmt->bind_param('ss', $month_like, $staff_id);
    $stmt->execute();
    $staff = $stmt->get_result()->fetch_assoc();
    if (!$staff) {
        $_SESSION['error'] = 'Staff not found or inactive.';
        header("Location: list.php");
        exit;
    }
} catch (Exception $e) {
    error_log("[$timestamp] Fetch staff failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch staff failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load staff details.';
    header("Location: list.php");
    exit;
}

// Handle payment submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_payment'])) {
    if ($_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = 'Invalid CSRF token.';
        header("Location: pay.php?staff_id=" . urlencode($staff_id));
        exit;
    }

    $amount = (float)$_POST['amount'];
    $payment_type = $_POST['payment_type'];
    $payment_date = $_POST['payment_date'];
    $bank_account_no = $_POST['bank_account_no'];
    $created_by = $_SESSION['user_id'] ?? $staff_id; // Fallback to staff_id if user_id is unset

    // Validate inputs
    if ($amount <= 0 || !in_array($payment_type, ['monthly', 'mid-month']) || empty($payment_date) || empty($bank_account_no)) {
        $_SESSION['error'] = 'Invalid payment details.';
        header("Location: pay.php?staff_id=" . urlencode($staff_id));
        exit;
    }

    // Check if payment exceeds remaining salary
    $remaining_salary = $staff['monthly_salary'] - ($staff['paid_this_month'] ?? 0) - ($staff['loan_deduction'] ?? 0);
    if ($amount > $remaining_salary) {
        $_SESSION['error'] = 'Payment amount exceeds remaining salary (KSh ' . number_format($remaining_salary, 2) . ').';
        header("Location: pay.php?staff_id=" . urlencode($staff_id));
        exit;
    }

    // Generate reference number
    $reference_no = 'PAY-' . date('YmdHis') . '-' . rand(1000, 9999);

    // Save payment
    $conn->begin_transaction();
    try {
        $stmt = $conn->prepare("
            INSERT INTO staff_payments (staff_id, branch_id, amount, payment_type, payment_date, bank_account_no, reference_no, created_by)
            SELECT ?, branch_id, ?, ?, ?, ?, ?, ? FROM staff WHERE staff_id = ?
        ");
        $stmt->bind_param('sdssssss', $staff_id, $amount, $payment_type, $payment_date, $bank_account_no, $reference_no, $created_by, $staff_id);
        $stmt->execute();

        // Deduct loan if applicable
        if ($staff['loan_deduction'] > 0 && $payment_type === 'monthly') {
            $stmt = $conn->prepare("
                INSERT INTO staff_payments (staff_id, branch_id, amount, payment_type, payment_date, bank_account_no, reference_no, created_by)
                SELECT ?, branch_id, ?, 'loan_repayment', ?, ?, ?, ? FROM staff WHERE staff_id = ?
            ");
            $loan_ref = 'LOAN-' . $reference_no;
            $stmt->bind_param('sdssssss', $staff_id, $staff['loan_deduction'], $payment_date, $bank_account_no, $loan_ref, $created_by, $staff_id);
            $stmt->execute();
        }

        $conn->commit();
        error_log("[$timestamp] Payment of KSh $amount ($payment_type) for staff ID $staff_id by $created_by");
        file_put_contents($log_file, "[$timestamp] Payment of KSh $amount ($payment_type) for staff ID $staff_id by $created_by\n", FILE_APPEND);
        $_SESSION['success'] = 'Payment processed successfully (Ref: ' . $reference_no . ').';
        header("Location: list.php");
        exit;
    } catch (Exception $e) {
        $conn->rollback();
        error_log("[$timestamp] Payment failed: " . $e->getMessage());
        file_put_contents($log_file, "[$timestamp] Payment failed: " . $e->getMessage() . "\n", FILE_APPEND);
        $_SESSION['error'] = 'Failed to process payment: ' . $e->getMessage();
        header("Location: pay.php?staff_id=" . urlencode($staff_id));
        exit;
    }
}

// Close connection
$conn->close();

// Include layout
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Process Payment - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); border: 1px solid #e0e0e0; }
        .btn-primary { background-color: #007bff; border-color: #007bff; transition: all 0.3s ease; }
        .btn-primary:hover { background-color: #0056b3; border-color: #0056b3; box-shadow: 0 2px 8px rgba(0,0,0,0.2); }
        .form-label { font-weight: 500; color: #333; margin-bottom: 8px; }
        .form-control, .form-select { border: 1px solid #007bff; border-radius: 5px; transition: border-color 0.3s ease; }
        .form-control:hover, .form-select:hover { border-color: #0056b3; }
        .alert { border-radius: 8px; }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="mb-4">Process Payment for <?php echo htmlspecialchars($staff['first_name'] . ' ' . $staff['last_name']); ?></h2>
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <div class="card">
            <div class="card-body">
                <h5>Payment Details</h5>
                <p><strong>Monthly Salary:</strong> KSh <?php echo number_format($staff['monthly_salary'], 2); ?></p>
                <p><strong>Paid This Month:</strong> KSh <?php echo number_format($staff['paid_this_month'] ?? 0, 2); ?></p>
                <p><strong>Loan Deduction:</strong> KSh <?php echo number_format($staff['loan_deduction'] ?? 0, 2); ?></p>
                <p><strong>Remaining Payable:</strong> KSh <?php echo number_format($staff['monthly_salary'] - ($staff['paid_this_month'] ?? 0) - ($staff['loan_deduction'] ?? 0), 2); ?></p>
                <form method="POST">
                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                    <div class="mb-3">
                        <label for="payment_type" class="form-label">Payment Type</label>
                        <select name="payment_type" id="payment_type" class="form-select" required>
                            <option value="monthly">Monthly Salary</option>
                            <option value="mid-month">Mid-Month Advance</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label for="amount" class="form-label">Amount (KES)</label>
                        <input type="number" step="0.01" name="amount" id="amount" class="form-control" required>
                    </div>
                    <div class="mb-3">
                        <label for="payment_date" class="form-label">Payment Date</label>
                        <input type="date" name="payment_date" id="payment_date" class="form-control" value="<?php echo date('Y-m-d'); ?>" required>
                    </div>
                    <div class="mb-3">
                        <label for="bank_account_no" class="form-label">Bank Account No</label>
                        <input type="text" name="bank_account_no" id="bank_account_no" class="form-control" value="<?php echo htmlspecialchars($staff['bank_account_no']); ?>" required>
                    </div>
                    <button type="submit" name="submit_payment" class="btn btn-primary"><i class="fas fa-money-bill"></i> Process Payment</button>
                    <a href="list.php" class="btn btn-secondary">Cancel</a>
                </form>
            </div>
        </div>
    </div>

    <?php include '../../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>