<?php
require_once __DIR__ . '/config/db.php';

$photoDir = __DIR__ . '/Uploads/staff_photos/';
$photoUrlBase = '/smartpay360/uploads/staff_photos/';
$logoUrl = '/smartpay360/assets/delta_logo.png'; // Replace with actual logo path
$companyDetails = [
    'name' => 'Delta Superstores Ltd',
    'address' => 'PO Box 215-30301, Nandi Hills Town',
    'email' => 'supermarketdelta@gmail.com'
];

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = __DIR__ . '/logs';
$log_file = "$log_dir/portfolio_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Auto-clean filenames: remove spaces from .jpg files
$rawPhotos = glob($photoDir . '*.jpg');
$photos = [];

foreach ($rawPhotos as $path) {
    $filename = basename($path);
    $cleaned = preg_replace('/\s+/', '', $filename);

    if ($filename !== $cleaned) {
        if (rename($path, $photoDir . $cleaned)) {
            error_log("[$timestamp] Renamed photo: $filename to $cleaned");
            file_put_contents($log_file, "[$timestamp] Renamed photo: $filename to $cleaned\n", FILE_APPEND);
        } else {
            error_log("[$timestamp] Failed to rename photo: $filename");
            file_put_contents($log_file, "[$timestamp] Failed to rename photo: $filename\n", FILE_APPEND);
        }
        $filename = $cleaned;
    }

    if (str_starts_with($filename, 'nid_') && preg_match('/nid_(\d+)\.jpg$/i', $filename)) {
        $photos[] = $photoDir . $filename;
    }
}

// Build staff map by national_id for active staff only
$staffMap = [];
try {
    $res = $conn->query("
        SELECT s.staff_id, s.first_name, s.last_name, s.national_id, s.role, b.branch_name
        FROM staff s
        LEFT JOIN branches b ON s.branch_id = b.id
        WHERE s.status = 'active'
    ");
    if (!$res) {
        throw new Exception("Query failed: " . $conn->error);
    }
    while ($row = $res->fetch_assoc()) {
        $national_id = trim($row['national_id']);
        if (preg_match('/^\d{7,9}$/', $national_id)) {
            $staffMap[$national_id] = $row;
        } else {
            error_log("[$timestamp] Invalid national_id in portfolio: $national_id");
            file_put_contents($log_file, "[$timestamp] Invalid national_id in portfolio: $national_id\n", FILE_APPEND);
        }
    }
    error_log("[$timestamp] Fetched " . count($staffMap) . " active staff records");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($staffMap) . " active staff records\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch staff failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch staff failed: " . $e->getMessage() . "\n", FILE_APPEND);
}

// Close connection
$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Staff Portfolio - Delta Superstores Ltd</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body { background-color: #f8f9fa; font-family: 'Arial', sans-serif; }
        .portfolio-header { 
            background: linear-gradient(to right, #007bff, #00aaff); 
            color: white; 
            padding: 2rem; 
            text-align: center; 
            border-radius: 10px; 
            margin-bottom: 2rem; 
        }
        .portfolio-header h1 { font-size: 2.5rem; font-weight: 700; }
        .portfolio-header p { font-size: 1.2rem; }
        .staff-grid { 
            display: grid; 
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); 
            gap: 20px; 
        }
        .staff-card { 
            background: white; 
            border: none; 
            border-radius: 12px; 
            box-shadow: 0 4px 12px rgba(0,0,0,0.1); 
            overflow: hidden; 
            transition: transform 0.3s ease, box-shadow 0.3s ease; 
        }
        .staff-card:hover { 
            transform: translateY(-5px); 
            box-shadow: 0 8px 20px rgba(0,0,0,0.15); 
        }
        .staff-photo { 
            width: 100%; 
            height: 200px; 
            object-fit: cover; 
            border-bottom: 3px solid #007bff; 
        }
        .staff-photo-placeholder { 
            width: 100%; 
            height: 200px; 
            background: #e9ecef; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            color: #6c757d; 
            font-size: 1.2rem; 
            border-bottom: 3px solid #007bff; 
        }
        .staff-info { 
            padding: 1.5rem; 
            text-align: center; 
        }
        .staff-info h5 { 
            font-size: 1.3rem; 
            font-weight: 600; 
            margin-bottom: 0.5rem; 
            color: #333; 
        }
        .staff-info p { 
            font-size: 0.95rem; 
            color: #6c757d; 
            margin-bottom: 0.3rem; 
        }
        .btn-edit, .btn-print { 
            border: none; 
            border-radius: 8px; 
            padding: 0.5rem 1rem; 
            font-size: 0.9rem; 
            margin: 0.3rem; 
            transition: background 0.3s ease; 
        }
        .btn-edit { background: #007bff; }
        .btn-edit:hover { background: #0056b3; }
        .btn-print { background: #28a745; }
        .btn-print:hover { background: #218838; }
        .btn-print-all { 
            background: linear-gradient(to right, #007bff, #00aaff); 
            border: none; 
            border-radius: 8px; 
            padding: 0.75rem 1.5rem; 
            font-size: 1rem; 
            margin-bottom: 1.5rem; 
        }
        .btn-print-all:hover { 
            background: linear-gradient(to right, #0056b3, #0088cc); 
        }
        .alert { 
            border-radius: 8px; 
            font-size: 0.9rem; 
        }
        .footer { 
            text-align: center; 
            margin-top: 3rem; 
            font-size: 0.9rem; 
            color: #888; 
        }
        /* Badge Styling */
        .badge-container { 
            display: none; /* Hidden on screen */
            width: 85.6mm; /* 3.37in */
            height: 54mm; /* 2.13in */
            border: 1px solid #000; 
            border-radius: 8px; 
            overflow: hidden; 
            margin: 10px; 
            font-family: Arial, sans-serif; 
            background: #fff; 
            box-shadow: 0 2px 5px rgba(0,0,0,0.1); 
            position: relative; 
        }
        .badge-header { 
            background: linear-gradient(to right, #007bff, #00aaff); 
            color: white; 
            padding: 5px; 
            text-align: center; 
            font-size: 12px; 
            font-weight: bold; 
        }
        .badge-body { 
            display: flex; 
            padding: 5px; 
            height: calc(100% - 25px); 
        }
        .badge-left { 
            width: 35mm; 
            display: flex; 
            flex-direction: column; 
            align-items: center; 
        }
        .badge-photo { 
            width: 30mm; 
            height: 35mm; 
            object-fit: cover; 
            border: 1px solid #ccc; 
            border-radius: 4px; 
            margin-bottom: 5px; 
        }
        .badge-photo-placeholder { 
            width: 30mm; 
            height: 35mm; 
            background: #e9ecef; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            border: 1px solid #ccc; 
            border-radius: 4px; 
            margin-bottom: 5px; 
            font-size: 8px; 
            color: #6c757d; 
        }
        .badge-qr { 
            width: 15mm; 
            height: 15mm; 
            margin-top: 5px; 
        }
        .badge-right { 
            flex: 1; 
            padding: 5px; 
            font-size: 9px; 
            line-height: 1.3; 
        }
        .badge-info p { 
            margin: 0 0 4px; 
            color: #333; 
        }
        .badge-info strong { 
            color: #007bff; 
        }
        .badge-logo { 
            width: 20mm; 
            height: 10mm; 
            object-fit: contain; 
            margin-top: 5px; 
        }
        .badge-footer { 
            position: absolute; 
            bottom: 2px; 
            left: 5px; 
            right: 5px; 
            font-size: 6px; 
            color: #555; 
            text-align: center; 
            line-height: 1.2; 
        }
        /* Print Styles */
        @media print {
            body, .container, .portfolio-header, .staff-grid, .footer, .btn-print-all { 
                display: none !important; 
            }
            .badge-container { 
                display: block !important; 
                page-break-inside: avoid; 
                margin: 0; 
                box-shadow: none; 
            }
            @page { 
                size: A4; 
                margin: 10mm; 
            }
            .badge-qr img { 
                width: 15mm !important; 
                height: 15mm !important; 
            }
        }
        @media (max-width: 576px) {
            .staff-grid { 
                grid-template-columns: 1fr; 
            }
            .portfolio-header h1 { 
                font-size: 2rem; 
            }
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <div class="portfolio-header">
            <h1>Delta Superstores Ltd Team</h1>
            <p>Meet the dedicated and active members of our family</p>
        </div>

        <div class="text-center">
            <button class="btn btn-print-all" onclick="printAllBadges()">
                <i class="fas fa-print me-2"></i> Print All Badges
            </button>
        </div>

        <?php if (empty($staffMap)): ?>
            <div class="alert alert-warning text-center">
                <i class="fas fa-exclamation-circle me-2"></i>No active staff members found.
            </div>
        <?php else: ?>
            <div class="staff-grid">
                <?php foreach ($photos as $fullPath):
                    $filename = trim(basename($fullPath));
                    $url = $photoUrlBase . $filename;
                    preg_match('/nid_(\d+)\.jpg$/i', $filename, $match);
                    $nid = $match[1] ?? null;
                    $staff = $nid && isset($staffMap[$nid]) ? $staffMap[$nid] : null;

                    if (!$staff) {
                        error_log("[$timestamp] No active staff match for photo NID: " . ($nid ?? 'unknown'));
                        file_put_contents($log_file, "[$timestamp] No active staff match for photo NID: " . ($nid ?? 'unknown') . "\n", FILE_APPEND);
                        continue;
                    }
                ?>
                    <div class="staff-card">
                        <?php if (file_exists($fullPath)): ?>
                            <img src="<?= htmlspecialchars($url) ?>?v=<?= time(); ?>" alt="<?= htmlspecialchars($staff['first_name'] . ' ' . $staff['last_name']) ?>" class="staff-photo">
                        <?php else: ?>
                            <div class="staff-photo-placeholder">No Photo</div>
                        <?php endif; ?>
                        <div class="staff-info">
                            <h5><?= htmlspecialchars($staff['first_name'] . ' ' . $staff['last_name']) ?></h5>
                            <p><strong>Role:</strong> <?= htmlspecialchars(ucfirst($staff['role'] ?: 'N/A')) ?></p>
                            <p><strong>Branch:</strong> <?= htmlspecialchars($staff['branch_name'] ?: 'N/A') ?></p>
                            <p><strong>ID:</strong> <?= htmlspecialchars($staff['staff_id']) ?></p>
                            <div>
                                <a href="/smartpay360/modules/staff/edit.php?national_id=<?= urlencode($staff['national_id']) ?>" class="btn btn-edit">
                                    <i class="fas fa-edit me-1"></i> Edit Profile
                                </a>
                                <button class="btn btn-print" onclick="printBadge('badge-<?= htmlspecialchars($staff['national_id']) ?>')">
                                    <i class="fas fa-print me-1"></i> Print Badge
                                </button>
                            </div>
                        </div>
                    </div>
                    <!-- Badge for Printing -->
                    <div class="badge-container" id="badge-<?= htmlspecialchars($staff['national_id']) ?>">
                        <div class="badge-header">Delta Superstores Ltd </div>
                        <div class="badge-body">
                            <div class="badge-left">
                                <?php if (file_exists($fullPath)): ?>
                                    <img src="<?= htmlspecialchars($url) ?>?v=<?= time(); ?>" class="badge-photo">
                                <?php else: ?>
                                    <div class="badge-photo-placeholder">No Photo</div>
                                <?php endif; ?>
                                <div class="badge-qr" id="qr-<?= htmlspecialchars($staff['national_id']) ?>" data-nid="<?= htmlspecialchars($staff['national_id']) ?>"></div>
                            </div>
                            <div class="badge-right">
                                <div class="badge-info">
                                    <p><strong>Name:</strong> <?= htmlspecialchars($staff['first_name'] . ' ' . $staff['last_name']) ?></p>
                                    <p><strong>Role:</strong> <?= htmlspecialchars(ucfirst($staff['role'] ?: 'N/A')) ?></p>
                                    <p><strong>Branch:</strong> <?= htmlspecialchars($staff['branch_name'] ?: 'N/A') ?></p>
                                    <p><strong>National ID:</strong> <?= htmlspecialchars($staff['national_id']) ?></p>
                                    <p><strong>Staff ID:</strong> <?= htmlspecialchars($staff['staff_id']) ?></p>
                                </div>
                                <img src="<?= htmlspecialchars($logoUrl) ?>" class="badge-logo" alt="Delta Superstores Logo">
                            </div>
                        </div>
                        <div class="badge-footer">
                            <?= htmlspecialchars($companyDetails['name']) ?>, <?= htmlspecialchars($companyDetails['address']) ?><br>
                            Email: <?= htmlspecialchars($companyDetails['email']) ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <div class="footer">
        © <?= date("Y"); ?> Delta Superstores Ltd. All rights reserved.
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/qrcode@1.5.1/build/qrcode.min.js"></script>
    <script>
        // Generate QR codes for each badge
        document.querySelectorAll('.badge-qr').forEach(qrDiv => {
            const nid = qrDiv.dataset.nid;
            new QRCode(qrDiv, {
                text: nid,
                width: 60, // 15mm at 300 DPI (~4px/mm)
                height: 60,
                colorDark: "#000000",
                colorLight: "#ffffff",
                correctLevel: QRCode.CorrectLevel.M
            });
        });

        function printBadge(badgeId) {
            const badge = document.getElementById(badgeId);
            const printWindow = window.open('', '_blank');
            printWindow.document.write(`
                <html>
                    <head>
                        <title>Print Staff Badge</title>
                        <style>
                            @page { size: 85.6mm 54mm; margin: 0; }
                            body { margin: 0; }
                            .badge-container {
                                width: 85.6mm;
                                height: 54mm;
                                border: 1px solid #000;
                                border-radius: 8px;
                                overflow: hidden;
                                font-family: Arial, sans-serif;
                                background: #fff;
                                position: relative;
                            }
                            .badge-header {
                                background: linear-gradient(to right, #007bff, #00aaff);
                                color: white;
                                padding: 5px;
                                text-align: center;
                                font-size: 12px;
                                font-weight: bold;
                            }
                            .badge-body {
                                display: flex;
                                padding: 5px;
                                height: calc(100% - 25px);
                            }
                            .badge-left {
                                width: 35mm;
                                display: flex;
                                flex-direction: column;
                                align-items: center;
                            }
                            .badge-photo {
                                width: 30mm;
                                height: 35mm;
                                object-fit: cover;
                                border: 1px solid #ccc;
                                border-radius: 4px;
                                margin-bottom: 5px;
                            }
                            .badge-photo-placeholder {
                                width: 30mm;
                                height: 35mm;
                                background: #e9ecef;
                                display: flex;
                                align-items: center;
                                justify-content: center;
                                border: 1px solid #ccc;
                                border-radius: 4px;
                                margin-bottom: 5px;
                                font-size: 8px;
                                color: #6c757d;
                            }
                            .badge-qr img {
                                width: 15mm;
                                height: 15mm;
                            }
                            .badge-right {
                                flex: 1;
                                padding: 5px;
                                font-size: 9px;
                                line-height: 1.3;
                            }
                            .badge-info p {
                                margin: 0 0 4px;
                                color: #333;
                            }
                            .badge-info strong {
                                color: #007bff;
                            }
                            .badge-logo {
                                width: 20mm;
                                height: 10mm;
                                object-fit: contain;
                                margin-top: 5px;
                            }
                            .badge-footer {
                                position: absolute;
                                bottom: 2px;
                                left: 5px;
                                right: 5px;
                                font-size: 6px;
                                color: #555;
                                text-align: center;
                                line-height: 1.2;
                            }
                        </style>
                    </head>
                    <body>
                        ${badge.outerHTML}
                    </body>
                </html>
            `);
            printWindow.document.close();
            printWindow.print();
            printWindow.close();
            <?php
            error_log("[$timestamp] Printed badge for national_id: " . ($staff['national_id'] ?? 'unknown'));
            file_put_contents($log_file, "[$timestamp] Printed badge for national_id: " . ($staff['national_id'] ?? 'unknown') . "\n", FILE_APPEND);
            ?>
        }

        function printAllBadges() {
            const badges = document.querySelectorAll('.badge-container');
            if (badges.length === 0) {
                alert('No badges to print.');
                return;
            }
            const printWindow = window.open('', '_blank');
            printWindow.document.write(`
                <html>
                    <head>
                        <title>Print All Staff Badges</title>
                        <style>
                            @page { size: A4; margin: 10mm; }
                            body { margin: 0; display: grid; grid-template-columns: repeat(2, 85.6mm); gap: 10mm; }
                            .badge-container {
                                width: 85.6mm;
                                height: 54mm;
                                border: 1px solid #000;
                                border-radius: 8px;
                                overflow: hidden;
                                font-family: Arial, sans-serif;
                                background: #fff;
                                page-break-inside: avoid;
                                position: relative;
                            }
                            .badge-header {
                                background: linear-gradient(to right, #007bff, #00aaff);
                                color: white;
                                padding: 5px;
                                text-align: center;
                                font-size: 12px;
                                font-weight: bold;
                            }
                            .badge-body {
                                display: flex;
                                padding: 5px;
                                height: calc(100% - 25px);
                            }
                            .badge-left {
                                width: 35mm;
                                display: flex;
                                flex-direction: column;
                                align-items: center;
                            }
                            .badge-photo {
                                width: 30mm;
                                height: 35mm;
                                object-fit: cover;
                                border: 1px solid #ccc;
                                border-radius: 4px;
                                margin-bottom: 5px;
                            }
                            .badge-photo-placeholder {
                                width: 30mm;
                                height: 35mm;
                                background: #e9ecef;
                                display: flex;
                                align-items: center;
                                justify-content: center;
                                border: 1px solid #ccc;
                                border-radius: 4px;
                                margin-bottom: 5px;
                                font-size: 8px;
                                color: #6c757d;
                            }
                            .badge-qr img {
                                width: 15mm;
                                height: 15mm;
                            }
                            .badge-right {
                                flex: 1;
                                padding: 5px;
                                font-size: 9px;
                                line-height: 1.3;
                            }
                            .badge-info p {
                                margin: 0 0 4px;
                                color: #333;
                            }
                            .badge-info strong {
                                color: #007bff;
                            }
                            .badge-logo {
                                width: 20mm;
                                height: 10mm;
                                object-fit: contain;
                                margin-top: 5px;
                            }
                            .badge-footer {
                                position: absolute;
                                bottom: 2px;
                                left: 5px;
                                right: 5px;
                                font-size: 6px;
                                color: #555;
                                text-align: center;
                                line-height: 1.2;
                            }
                        </style>
                    </head>
                    <body>
                        ${Array.from(badges).map(b => b.outerHTML).join('')}
                    </body>
                </html>
            `);
            printWindow.document.close();
            printWindow.print();
            printWindow.close();
            <?php
            error_log("[$timestamp] Printed all badges for " . count($staffMap) . " active staff");
            file_put_contents($log_file, "[$timestamp] Printed all badges for " . count($staffMap) . " active staff\n", FILE_APPEND);
            ?>
        }
    </script>
</body>
</html>